/*
 * netClient.h - defines the structures and types which are used for
 *	communication between generic network code and the code for a
 *	custom client type.
 *	- Chris Gray, cg@ami-cg.GraySage.Edmonton.AB.CA, Nov 1998.
 */

/*
 * This is the structure that is used by the generic net code to keep
 * track of clients. It is private to the generic net code, but the
 * custom client code is passed a pointer to it at nc_alloc time, and must
 * pass that same pointer back to all generic net routines it calls on.
 */

struct GenericClient;

/*
 * Request_t is a simple, generic request type. It is used so that
 * output data can be queued in the server when sockets block.
 */

typedef struct Request {
    struct Request *rq_next;
    unsigned long int rq_id;
    unsigned short int rq_type;
    unsigned short int rq_availLen;
    unsigned short int rq_usedLen;
    union {
	byte_t ru_bytes[2];
	char ru_text[2];
    } rq_u;
} Request_t;

/* These fields are:

    rq_next - pointer to next Request_t in any queue.
    rq_id - 32 bit unique id for this active client connection. This can
	be used inside a multiplexor program, which would speak to the
	server using a binary protocol, and would need this id to send
	data to the proper real client.
    rq_type - the type of this Request_t. Not used in this interface.
    rq_availLen - the amount of space actually available in this Request_t's
	rq_data.
    rq_usedLen - the amount of data actually in this Request_t's rq_data.
    rq_data - the actual data. This is a varying size amount of stuff,
	defined by rq_availLen.

   The only place we see Request_t's in this interface is when they are
   holding text that is to be sent out. Code for a custom client that
   is doing a binary interface to a remote custom client might care about
   the type, and that remote client certainly would. For example, the
   nc_setPrompt call, as handled by such custom client code, might use
   the rq_type field to indicate that the request contains a new prompt
   to be displayed by the custom client.
 */

/*
 * NetGeneric_t is a structure containing function pointers which define
 * the set of facilities which custom client type code can call on
 * in the generic net code.
 */

typedef struct NetGeneric {
    int (*ng_netWrite)(struct GenericClient *gcl, const void *buffer, int len);
    void (*ng_addToWriteWaiters)(struct GenericClient *gcl);
    void (*ng_removeFromWriteWaiters)(struct GenericClient *gcl);
    void (*ng_setWindowSize)(struct GenericClient *gcl, int rows, int cols);
    void (*ng_getWindowSize)(struct GenericClient *gcl, int *pRows,int *pCols);
    void (*ng_setTerminalType)(struct GenericClient *gcl,
			       const char *terminalType);
    const char *(*ng_getTerminalType)(struct GenericClient *gcl);
    int (*ng_netRead)(struct GenericClient *gcl, void *buffer, int len);
    void (*ng_handleLine)(struct GenericClient *gcl, char *buffer, int len);
    void (*ng_freeRequest)(Request_t *rq);
    void (*ng_killClient)(struct GenericClient *gcl);
} NetGeneric_t;

/* These entry points into the generic net code are:

    ng_netWrite - raw write of data over the socket. Return -1 if error,
	or the number of bytes actually written.
    ng_addToWriteWaiters - we wanted to write to the socket, but 'netWrite'
	didn't report writing it all. So, we assume the socket is blocked,
	and request notification, via 'nc_write', of when it unblocks.
    ng_removeFromWriteWaiters - we no longer have anything left to send out,
	so we don't need unblock notification any more (this is the default
	state).
    ng_setWindowSize - we have just received information about a change (or
	the initial value of) the output text window size. Inform the generic
	code.
    ng_setTerminalType - we have just received information about the type of
	the terminal the user is using. Inform the generic code.
    ng_netRead - raw read of data from the socket. Return -1 if error, or the
	number of bytes actually read.
    ng_freeRequest - a request (containing text) has been fully sent, and we
	no longer need it. Return it to the rest of the system.
    ng_killClient - some client-type code has determined that the connection
	should go away. Make it so.
 */

/*
 * NetClient_t is a structure of function pointers which defines the
 * interface that code for a custom client type should export.
 */

typedef struct NetClient {
    const void *(*nc_alloc)(struct GenericClient *gcl,
			    const NetGeneric_t *ng, int auxData);
    void (*nc_setBlank)(void *clientData, bool_t blankOn);
    void (*nc_setPrompt)(void *clientData, const char *newPrompt);
    void (*nc_newName)(void *clientData, const char *newName);
    void (*nc_putText)(void *clientData, Request_t *rq);
    void (*nc_flush)(void *clientData);
    void (*nc_read)(void *clientData);
    void (*nc_write)(void *clientData);
    void (*nc_free)(void *clientData);
} NetClient_t;

/* These entry points into the client-type code are:

    nc_alloc - called to create a new client of this type. 'auxData' is
	whatever might be useful. For telnet it is initial flags. This call
	should return a pointer to the custom client type code's client
	data structure.
    nc_setBlank - whether or not we want intput blanked right now. Turning it
	on is done when getting passwords.
    nc_setPrompt - set a new prompt for the user
    nc_newName - a new character name (useful when logging)
    nc_putText - this request structure contains some text that should
	be sent to the client. It is a queue-able request.
    nc_flush - we should try now to flush out all stuff we have queued
	to go out. This usually happens when the server is now switching
	to sending output to some other user. We want to buffer stuff up
	in the first place to cut down on writes to the socket.
    nc_read - this client now has data available on the socket, and should
	read it.
    nc_write - this client now can write to its socket, and should do so
	(these only happen if the custom client code has requested them)
    nc_free - delete this custom client structure, we are done with it.
 */
