/*
 * socket.c - code for dealing with sockets. Uses 'select'. Does non-blocking
 *	sockets for both reading and writing. Registers multiple possible
 *	client-types that do further processing (e.g. telnet).
 *	- Chris Gray, cg@ami-cg.GraySage.Edmonton.AB.CA, Nov/Dec 1998.
 */

/*
 * This code is provided as-is. No warranty of any kind is given.
 */

/*
 * As of this writing, only compilation under Linux has been tested.
 * More will come.
 */

#if defined(__unix) || defined(AMIGA)

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#if defined(AIX)
#include <sys/select.h>
#endif
#if defined(SOLARIS)
#include <sys/filio.h>
#endif
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>

typedef int Socket_t;
#define INVALID_SOCKET	-1

#elif defined(_WIN32)

/* Can redefine this before including the header file, to change the
   number of sockets we can have at once in a select call. 64 is
   the default. */

#define FD_SETSIZE	64

#include <winsock2.h>
#include <wininet.h>

typedef SOCKET Socket_t;

#else
%%%% No environment specified

#endif

#if defined(AMIGA)
#include <proto/dos.h>
extern int fcntl(int fd, int cmd, int arg);
#endif

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>

#include "../include/devmud.h"
#include "../include/types.h"
#include "../include/telnetFlags.h"
#include "../include/netClient.h"

#define ERRORS_TO_STDERR	1
#define DEBUG_ENABLED		0
#define DO_ABORTS		0

#define MAX_DELAY	1000

/*
 * This type is used to keep track of the kinds of custom client types
 * that have been registered with us. Until we get at least one, we
 * won't even have an sockets to work with.
 */

typedef struct ClientType {
    struct ClientType *clt_next;
    struct ClientType *clt_prev;
    NetClient_t *clt_netClient;
    unsigned short int clt_port;
    int clt_auxData;
    int clt_socket;
} ClientType_t;

/*
 * The actual definition of the GenericClient structure.
 */

typedef struct GenericClient {
    struct GenericClient *gcl_next;
    struct GenericClient *gcl_prev;
    const ClientType_t *gcl_type;
    const void *gcl_clientData;
    const void *gcl_worldData;
    char *gcl_terminalType;
    int gcl_windowRows;
    int gcl_windowCols;
    Socket_t gcl_socket;
    bool_t gcl_isShutDown;
    bool_t gcl_isDeleted;
} GenericClient_t;

/* Structure for a time-delay action. */

typedef struct action {
    struct action *ac_next;
    int ac_delay;
    void (*ac_function)(void *worldData, void *arg1, int arg2);
    void *ac_worldData;
    void *ac_arg1;
    int ac_arg2;
} Action_t;

/* World functions we need to reference */

static void *(*NewClient)(GenericClient_t *gcl);
static void (*HandleLine)(const void *cl, char *buffer, int len);
static void (*ClientGone)(void *cl);

static ClientType_t *ClientTypes;	/* list of custom client types */
static fd_set AllSocketBits;		/* all current sockets */
static fd_set ClientWriteBits;		/* all write-blocked sockets */
static Socket_t MainSocketMax;		/* max of just accept sockets */
static Socket_t MaxFd;			/* largest, used to bound select */
static GenericClient_t *Clients;	/* list of all clients */
static unsigned long int NextId;	/* next unique client ID */
static bool_t QuitRequested;
static NetGeneric_t TheNetGeneric;	/* the only one of these. */

static Action_t *Actions;
int CurrentDelay, LastTime;

#if defined(AMIGA)
static long int MyAddr;
#elif defined(_WIN32)
static unsigned int MyAddr;
#else
static int MyAddr;
#endif

#if defined(_WIN32)
static WSADATA WSAData;
#endif

#if DEBUG_ENABLED
#define DEBUG(x)    printf x
#else
#define DEBUG(x)
#endif

/*
 * myPerror - print to somewhere, an error regarding some operation.
 *	Note that all of our calls are from socket call errors.
 */

#define ERROR_BUFF_LEN	1000

#if defined(__unix) || defined(AMIGA)

static void
myPerror(const char *message)
{
    char buffer[ERROR_BUFF_LEN];
    char *reason;

#if defined(SUNOS4)
    extern int sys_nerr;
    extern char *sys_errlist[];

    if (errno >= sys_nerr) {
	reason = NULL;
    } else {
	reason = sys_errlist[errno];
    }
#else
    reason = strerror(errno);
#endif
    if (reason == NULL) {
	sprintf(&buffer[0], "%s: unknown error %d", message, errno);
    } else {
	sprintf(&buffer[0], "%s: %s", message, reason);
    }
#if ERRORS_TO_STDERR
    fprintf(stderr, "%s\n", &buffer[0]);
#else
    %%%%
#endif
}

#elif defined(_WIN32)

static void
myPerror(const char *message)
{
    HMODULE wsMod;
    DWORD mLen, len, errorCode;
    char buffer[ERROR_BUFF_LEN];

    errorCode = WSAGetLastError();
    sprintf(&buffer[0], "%s: ", message);
    mLen = strlen(&buffer[0]);
    wsMod = GetModuleHandle("wsock32");
    len =
	FormatMessage(FORMAT_MESSAGE_FROM_HMODULE | FORMAT_MESSAGE_FROM_SYSTEM,
		      (LPCVOID) wsMod, errorCode, 0, &buffer[mLen],
		      ERROR_BUFF_LEN - mLen - 1, NULL);
    if (len == 0) {
	sprintf(&buffer[mLen], "FormatMessage failed reason %d, orig error %d",
		GetLastError(), errorCode);
    } else {
	/* trim any trailing CR/LF */
	mLen = strlen(&buffer[0]);
	if (buffer[mLen - 1] == '\n') {
	    buffer[mLen - 1] = '\0';
	    mLen -= 1;
	}
	if (buffer[mLen - 1] == '\r') {
	    buffer[mLen - 1] = '\0';
	}
    }
#if ERRORS_TO_STDERR
    fprintf(stderr, "%s\n", &buffer[0]);
#else
    %%%%
#endif
}

#undef ERROR_BUFF_LEN

#else
%%%%
#endif

/*
 * killSocket - get rid of a socket.
 */

static void
killSocket(Socket_t s)
{

    shutdown(s, 2);
#if defined(_WIN32)
    closesocket(s);
#elif defined(AMIGA)
    CloseSocket(s);
#else
    close(s);
#endif
}

/*
 * deleteClient - delete this client. We are done with it.
 */

static void
deleteClient(GenericClient_t *gcl)
{

    (*gcl->gcl_type->clt_netClient->nc_free)((void *) gcl->gcl_clientData);
    if (gcl->gcl_prev != NULL) {
	gcl->gcl_prev->gcl_next = gcl->gcl_next;
    } else {
	Clients = gcl->gcl_next;
    }
    if (gcl->gcl_next != NULL) {
	gcl->gcl_next->gcl_prev = gcl->gcl_prev;
    }
    if (gcl->gcl_terminalType != NULL) {
	free(gcl->gcl_terminalType);
    }
    free(gcl);
}

/*
 * detachSocket - detach the client from the socket.
 */

static void
detachSocket(GenericClient_t *gcl)
{
    Socket_t s, max;

    s = gcl->gcl_socket;
    if (s != INVALID_SOCKET) {
	gcl->gcl_socket = INVALID_SOCKET;
	FD_CLR(s, &AllSocketBits);
	FD_CLR(s, &ClientWriteBits);
	killSocket(s);
	if (s == MaxFd) {
	    /* This was the biggest, and is now gone. Find new biggest. */
	    max = MainSocketMax;
	    gcl = Clients;
	    while (gcl != NULL) {
		if (gcl->gcl_socket != INVALID_SOCKET && gcl->gcl_socket > max)
		{
		    max = gcl->gcl_socket;
		}
		gcl = gcl->gcl_next;
	    }
	    MaxFd = max;
	}
    }
}

/*
 * abortClient - do everything needed to make a client go away.
 */

static void
abortClient(GenericClient_t *gcl)
{

    DEBUG(("abortClient\n"));
    if (! gcl->gcl_isShutDown) {
	/* Avoid recursion here, just in case! */
	gcl->gcl_isShutDown = TRUE;
	if (! gcl->gcl_isDeleted) {
	    (*ClientGone)((void *) gcl->gcl_worldData);
	}
    }
    detachSocket(gcl);
    deleteClient(gcl);
}

/*
 * setupSocket - create a socket, and bind it to the given port.
 */

static bool_t
setupSocket(int *pSocket, unsigned short int port)
{
    struct sockaddr_in myAddr;
    Socket_t s;
    long int flag;

    /* Create a non-blocking socket */
    flag = 1;
#if defined(_WIN32)
    s = WSASocket(AF_INET, SOCK_STREAM, 0, NULL, 0, 0);
    if (s == INVALID_SOCKET) {
	myPerror("can't create server socket");
    }
    if (ioctlsocket(s, FIONBIO, (unsigned long *) &flag) == SOCKET_ERROR) {
	myPerror("ioctlsocket on server socket");
	killSocket(s);
	return FALSE;
    }
#else
    s = socket(AF_INET, SOCK_STREAM, 0);
    if (s < 0) {
	myPerror("can't create server socket");
	return FALSE;
    }
    if (ioctl(s, FIONBIO, (char *) &flag) == -1) {
	myPerror("ioctl on server socket");
	killSocket(s);
	return FALSE;
    }
#endif

    /* Do this *before* the bind/listen */
    if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR,(char *)&flag, sizeof(flag))<0){
	myPerror("setsockopt(SO_REUSEADDR)");
	killSocket(s);
	return FALSE;
    }

    /* Bind it to the required port. */
    memset(&myAddr, 0, sizeof(struct sockaddr_in));
    myAddr.sin_port = htons(port);
    myAddr.sin_family = AF_INET;
    if (bind(s, (struct sockaddr *) &myAddr, sizeof(struct sockaddr_in)) < 0) {
	myPerror("bind");
	killSocket(s);
	return FALSE;
    }

    /* Accept connections on it. */
    if (listen(s, 5) < 0) {
	myPerror("listen");
	killSocket(s);
	return FALSE;
    }

    /* Add it to the set of sockets our 'select' pays attention to. */
    FD_SET(s, &AllSocketBits);
    if (s > MainSocketMax) {
	MainSocketMax = s;
    }
    if (s > MaxFd) {
	MaxFd = s;
    }
    *pSocket = s;
    return TRUE;
}

/*
 * netWrite - write some data directly to a client socket.
 */

int
netWrite(GenericClient_t *gcl, const void *buf, int len)
{
    int res;

    res = send(gcl->gcl_socket, buf, len, 0);
    if (res < 0) {
#if defined(_WIN32)
	if (WSAGetLastError() != WSAEWOULDBLOCK)
#else
	if (errno != EWOULDBLOCK)
#endif
	{
	    /* Something wrong - shutdown the client. */
	    abortClient(gcl);
	} else {
	    /* Fake this, so caller will not think we have died. */
	    res = 0;
	}
    }
    return res;
}

/*
 * netRead - read some data directly from a client socket.
 */

int
netRead(GenericClient_t *gcl, void *buf, int len)
{
    int res;

    res = recv(gcl->gcl_socket, buf, len, 0);
    if (res <= 0) {
#if defined(_WIN32)
	if (WSAGetLastError() != WSAEWOULDBLOCK)
#else
	if (errno != EWOULDBLOCK)
#endif
	{
	    /* Something wrong with the client. */
	    abortClient(gcl);
	}
    }
    return res;
}

/*
 * addToWriteWaiters - we have filled up an outgoing socket, and must
 *	stop sending to it for a while. Add that socket to the set
 *	we select on for output ready.
 */

void
addToWriteWaiters(GenericClient_t *gcl)
{

    FD_SET(gcl->gcl_socket, &ClientWriteBits);
}

/*
 * removeFromWriteWaiters - this client no longer needs to be woken up
 *	when something can be sent on its socket.
 */

void
removeFromWriteWaiters(GenericClient_t *gcl)
{

    FD_CLR(gcl->gcl_socket, &ClientWriteBits);
}

/*
 * scanClients - scan the client sockets for something we can do.
 */

static void
scanClients(fd_set *readFds, fd_set *writeFds, fd_set *exceptFds)
{
    GenericClient_t *gcl, *gclTemp;

    gcl = Clients;
    while (gcl != NULL) {
	/* Any of the calls here can delete the client, so be careful. */
	gclTemp = gcl;
	gcl = gcl->gcl_next;
	if (FD_ISSET(gclTemp->gcl_socket, exceptFds)) {
	    abortClient(gclTemp);
	} else {
	    if (gclTemp->gcl_socket != INVALID_SOCKET &&
		FD_ISSET(gclTemp->gcl_socket, readFds))
	    {
		/* Something to read from the client socket. */
		(*gclTemp->gcl_type->clt_netClient->nc_read)
		    ((void *) gclTemp->gcl_clientData);
	    }
	    if (gclTemp->gcl_socket != INVALID_SOCKET &&
		FD_ISSET(gclTemp->gcl_socket, writeFds))
	    {
		/* We can write more to the client socket. */
		(*gclTemp->gcl_type->clt_netClient->nc_write)
		    ((void *) gclTemp->gcl_clientData);
	    }
	}
    }
}

/*
 * handleConnection - a new connection has come in on a main socket.
 */

static void
handleConnection(ClientType_t *clt)
{
    struct sockaddr_in from;
    GenericClient_t *gcl;
    Socket_t s;
    long int flag;
#if defined(AMIGA)
    LONG fromLen;
#elif defined(SOLARIS) || defined(AIX)
    int fromLen;
#else
    unsigned int fromLen;
#endif

    DEBUG(("handleConnection from port %d\n", clt->clt_port));
    fromLen = sizeof(struct sockaddr_in);
    s = accept(clt->clt_socket, (struct sockaddr *) &from, &fromLen);
#if defined(_WIN32)
    if (s == INVALID_SOCKET) {
	s = WSAGetLastError();
	if (s != WSAEINTR && s != WSAEWOULDBLOCK) {
	    myPerror("accept");
#if DO_ABORTS
	    myAbort("client socket error");
#else
	    return;
#endif
	}
    }
#else
    if (s < 0) {
	if (errno != EINTR && errno != EWOULDBLOCK) {
	    myPerror("accept");
#if DO_ABORTS
	    myAbort("client socket error");
#else
	    return;
#endif
	}
    }
#endif
    else {
	/* Set the new client socket to be non-blocking. */
	flag = 1;
#if defined(_WIN32)
	if (ioctlsocket(s, FIONBIO, (unsigned long *) &flag) == SOCKET_ERROR) {
	    myPerror("ioctlsocket on new client socket");
	    killSocket(s);
	    return;
	}
#else
	if (ioctl(s, FIONBIO, (char *) &flag) == -1) {
	    myPerror("ioctl on new client socket");
	    killSocket(s);
	    return;
	}
#endif

	/* If the socket is a local one, then turn off the "Nagle
	   Algorithm" on it. That improves performance (a lot!) for
	   the local socket. */
	if (from.sin_addr.s_addr == MyAddr &&
	    setsockopt(s, IPPROTO_TCP, TCP_NODELAY, (char *) &flag,
		       sizeof(int)) != 0)
	{
	    myPerror("setsockopt on new socket");
	    killSocket(s);
	    return;
	}

	gcl = malloc(sizeof(GenericClient_t));
	if (gcl == NULL) {
	    killSocket(s);
	    return;
	}
	gcl->gcl_type = clt;
	gcl->gcl_socket = s;
	gcl->gcl_windowRows = 0;
	gcl->gcl_windowCols = 0;
	gcl->gcl_terminalType = NULL;
	gcl->gcl_isShutDown = FALSE;
	gcl->gcl_isDeleted = FALSE;
	gcl->gcl_clientData =
	    (*clt->clt_netClient->nc_alloc)(gcl, &TheNetGeneric,
					    clt->clt_auxData);
	if (gcl->gcl_clientData == NULL) {
	    killSocket(s);
	    free(gcl);
	    return;
	}
	gcl->gcl_worldData = (*NewClient)(gcl);
	if (gcl->gcl_worldData == NULL) {
	    (*clt->clt_netClient->nc_free)((void *) gcl->gcl_clientData);
	    killSocket(s);
	    free(gcl);
	    return;
	}

	/* All setup. Add it to our list of clients. */
	gcl->gcl_next = Clients;
	gcl->gcl_prev = NULL;
	if (Clients != NULL) {
	    Clients->gcl_prev = gcl;
	}
	Clients = gcl;

	/* Add this new socket to the set we 'select' on, awaiting input
	   from the user at the other end of it. */
	FD_SET(s, &AllSocketBits);
	if (s > MaxFd) {
	    MaxFd = s;
	}
    }
}

/*
 * socket_addTimedAction - add a time-driven action.
 */

int
socket_addTimedAction(int delay, void (*function)(void *, void *, int),
		      void *worldData, void *arg1, int arg2)
{
    Action_t *ac, **acp;
    int now, delta;

    now = time(NULL);
    delta = now - LastTime;
    ac = Actions;
    while (ac != NULL) {
	ac->ac_delay -= delta;
	ac = ac->ac_next;
    }

    ac = (Action_t *) malloc(sizeof(Action_t));
    if (ac == NULL) {
	return -1;
    }

    ac->ac_delay = delay;
    ac->ac_function = function;
    ac->ac_worldData = worldData;
    ac->ac_arg1 = arg1;
    ac->ac_arg2 = arg2;
    if (Actions == NULL || delay < Actions->ac_delay) {
	CurrentDelay = delay;
	LastTime = now;
	ac->ac_next = Actions;
	Actions = ac;
    } else {
	acp = &Actions;
	while (*acp != NULL && (*acp)->ac_delay <= delay) {
	    acp = &(*acp)->ac_next;
	}
	ac->ac_next = *acp;
	*acp = ac;
    }
    return 0;
}

/*
 * socket_cancelActions - cancel all actions pending with the given arg1.
 */

void
socket_cancelActions(void *worldData)
{
    Action_t *ac, **acp;

    acp = &Actions;
    while (TRUE) {
	ac = *acp;
	if (ac == NULL) {
	    break;
	}
	if (ac->ac_worldData == worldData) {
	    *acp = ac->ac_next;
	    free(ac);
	} else {
	    acp = &ac->ac_next;
	}
    }
}

/*
 * doTimedActions - trigger any time-based actions that are ready.
 */

static void
doTimedActions(void)
{
    int now, delta;
    Action_t *ac;
    Action_t **acp;

    now = time(NULL);
    delta = now - LastTime;
    LastTime = now;
    ac = Actions;
    while (ac != NULL) {
	ac->ac_delay -= delta;
	ac = ac->ac_next;
    }
    acp = &Actions;
    while (TRUE) {
	ac = *acp;
	if (ac == NULL || ac->ac_delay > 0) {
	    break;
	}
	*acp = ac->ac_next;
	(*ac->ac_function)(ac->ac_worldData, ac->ac_arg1, ac->ac_arg2);
	free(ac);
    }
    if (Actions == NULL) {
	CurrentDelay = MAX_DELAY;
    } else {
	CurrentDelay = Actions->ac_delay;
    }
}

/*
 * setWindowSize - callable from client-type code to set the text output
 *	window size.
 */

static void
setWindowSize(GenericClient_t *gcl, int rows, int cols)
{

    gcl->gcl_windowRows = rows;
    gcl->gcl_windowCols = cols;
}

/*
 * setTerminalType - callable from client-type code to set the type of
 *	the terminal or client.
 */

static void
setTerminalType(GenericClient_t *gcl, const char *terminalType)
{

    if (gcl->gcl_terminalType != NULL) {
	free(gcl->gcl_terminalType);
    }
    gcl->gcl_terminalType = strdup(terminalType);
    /* May get NULL - that is OK. */
}

/*
 * socket_getWindowSize - export this to the custom types and generic code.
 */

void
socket_getWindowSize(GenericClient_t *gcl, int *pRows, int *pCols)
{

    *pRows = gcl->gcl_windowRows;
    *pCols = gcl->gcl_windowCols;
}

/*
 * socket_getTerminalType - export to custom clients and generic code.
 */

const char *
socket_getTerminalType(GenericClient_t *gcl)
{

    return gcl->gcl_terminalType;
}

/*
 * handleLine - bounce complete input lines through to the client.
 */

static void
handleLine(GenericClient_t *gcl, char *buffer, int len)
{

    (*HandleLine)(gcl->gcl_worldData, buffer, len);
}

/*
 * freeRequest - we provide this to client type code. It would normally
 *	call some real routine somewhere in the rest of the system.
 */

static void
freeRequest(Request_t *rq)
{

    free(rq);
}

#if defined(_WIN32)
#define MAXHOSTNAMELEN	INTERNET_MAX_HOST_NAME_LENGTH
#endif

/*
 * socket_killClient - force disconnection of the client.
 */

void
socket_killClient(GenericClient_t *gcl)
{

    gcl->gcl_isDeleted = TRUE;
}

/*
 * socket_initialize - initial setup code for this module.
 */

int
socket_initialize(void)
{
    char nameBuffer[MAXHOSTNAMELEN];
    struct hostent *hp;

#if defined(_WIN32)
    if (WSAStartup(MAKEWORD(1, 0), &WSAData) < 0) {
	/* WinSock startup error. */
	return 1;
    }
#endif

    /* Basic initialization */
    NewClient = NULL;
    HandleLine = NULL;
    ClientGone = NULL;
    ClientTypes = NULL;
    FD_ZERO(&AllSocketBits);
    FD_ZERO(&ClientWriteBits);
    MainSocketMax = 0;
    MaxFd = 0;
    NextId = 1;
    QuitRequested = FALSE;
    CurrentDelay = MAX_DELAY;
    LastTime = time(NULL);
    srand(LastTime);

    /* Find out the IP address of this machine. */
    if (gethostname(&nameBuffer[0], MAXHOSTNAMELEN) < 0) {
	myPerror("gethostname");
	return 1;
    }
    hp = gethostbyname(&nameBuffer[0]);
    if (hp == NULL) {
	myPerror("gethostbyname for self");
	return 1;
    }
#if defined(AMIGA)
    MyAddr = (long int) hp->h_addr;
#else
    MyAddr = (int) hp->h_addr;
#endif

    /* Setup the NetGeneric_t that we pass to client type code. */
    TheNetGeneric.ng_netWrite = netWrite;
    TheNetGeneric.ng_addToWriteWaiters = addToWriteWaiters;
    TheNetGeneric.ng_removeFromWriteWaiters = removeFromWriteWaiters;
    TheNetGeneric.ng_setWindowSize = setWindowSize;
    TheNetGeneric.ng_getWindowSize = socket_getWindowSize;
    TheNetGeneric.ng_setTerminalType = setTerminalType;
    TheNetGeneric.ng_getTerminalType = socket_getTerminalType;
    TheNetGeneric.ng_netRead = netRead;
    TheNetGeneric.ng_handleLine = handleLine;
    TheNetGeneric.ng_freeRequest = freeRequest;
    TheNetGeneric.ng_killClient = socket_killClient;

    return 0;
}

/*
 * socket_use_functions - extract any external function pointers we need.
 */

void
socket_use_functions(struct interface *it)
{

    while (it->name != NULL) {
	if (strcmp(it->name, "world_newClient") == 0) {
	    NewClient = (void *(*)(GenericClient_t *)) it->function;
	}
	if (strcmp(it->name, "world_handleLine") == 0) {
	    HandleLine = (void (*)(const void *, char *, int)) it->function;
	}
	if (strcmp(it->name, "world_clientGone") == 0) {
	    ClientGone = (void (*)(void *)) it->function;
	}
	++it;
    }
}

/*
 * interruptHandler - catch SIGINT signals. Tell main loop that we
 *	want it to go away eventually.
 */

static void
interruptHandler(int sig)
{

    QuitRequested = TRUE;
}

/*
 * socket_start - wait for and handle requests from clients. This is a
 *	"mainline" routine - it does not return until it is asked to,
 *	*and* there are no clients left.
 */

int
socket_start(void)
{
    ClientType_t *clt, *cltTemp;
    GenericClient_t *gcl, *gclTemp;
    Action_t *ac, *acTemp;
    fd_set readFds, writeFds, exceptFds;
    struct timeval timeval;
    int res;
    bool_t mainException;

    if (NewClient == NULL || HandleLine == NULL || ClientGone == NULL) {
	return 1;
    }

    if (signal(SIGINT, interruptHandler) == SIG_ERR) {
	return 1;
    }

    while (! QuitRequested || Clients != NULL) {
	/* Modify this timeval if this code is triggering events. */
	timeval.tv_sec = CurrentDelay;
	timeval.tv_usec = 0;
	readFds = AllSocketBits;
	writeFds = ClientWriteBits;
	exceptFds = readFds;
	DEBUG(("Going into select, MaxFd = %d, timeval = (%ld,%ld)\n",
	       MaxFd, timeval.tv_sec, timeval.tv_usec));

	res = select(MaxFd + 1, &readFds, &writeFds, &exceptFds, &timeval);

	DEBUG(("Got out of select, res = %d\n", res));
#if defined(_WIN32)
	if (res == SOCKET_ERROR)
#else
	if (res < 0)
#endif
	{
#if defined(_WIN32)
	    if (WSAGetLastError() == WSAEINTR)
#else
	    if (errno == EINTR)
#endif
	    {
		/* The select was interrupted. Assume it was an external
		   request for us to go away. */
		break;
	    } else {
		myPerror("select");
		break;
	    }
	} else if (res > 0) {

	    /* Activity on some socket. */
	    mainException = FALSE;
	    clt = ClientTypes;
	    while (clt != NULL) {
		if (FD_ISSET(clt->clt_socket, &exceptFds)) {
		    mainException = TRUE;
		    break;
		}
		clt = clt->clt_next;
	    }
	    if (mainException) {
		/* An exception on one of our main sockets. Bail out. */
		/* Better might be to try to shutdown that client type, but
		   that is harder if it has active clients. */
		break;
	    }

	    /* Check for a new connection on a main socket. */
	    clt = ClientTypes;
	    while (clt != NULL) {
		if (FD_ISSET(clt->clt_socket, &readFds)) {
		    handleConnection(clt);
		}
		clt = clt->clt_next;
	    }

	    /* Check for activity on the client sockets. */
	    scanClients(&readFds, &writeFds, &exceptFds);

	} else {
	    /* Nothing on any socket - timer has gone off. */
	    doTimedActions();
	}

	/* Delete the structures for any clients that we should delete. */
	gcl = Clients;
	while (gcl != NULL) {
	    gclTemp = gcl;
	    gcl = gcl->gcl_next;
	    if (gclTemp->gcl_isDeleted) {
		abortClient(gclTemp);
	    }
	}
    }

    gcl = Clients;
    while (gcl != NULL) {
	gclTemp = gcl;
	gcl = gcl->gcl_next;
	abortClient(gclTemp);
    }

    clt = ClientTypes;
    while (clt != NULL) {
	cltTemp = clt;
	clt = clt->clt_next;
	killSocket(cltTemp->clt_socket);
	free(cltTemp);
    }

    ac = Actions;
    while (ac != NULL) {
	acTemp = ac;
	ac = ac->ac_next;
	free(acTemp);
    }

#if defined(_WIN32)
    (void) WSACleanup();
#endif
    return 0;
}

/*
 * socket_stop - external code wants us to go away. Note that we are normally
 *	sitting inside 'socket_start', and will only leave that routine after
 *	being asked to (here), and when we have no clients left.
 */

void
socket_stop(void)
{

    QuitRequested = TRUE;
}

/*
 * socket_addClientType - this is the externally available interface by which
 *	other code can ask us to support a new type of client. They give
 *	us a vector of functions to call for various things, the port to
 *	use, and an int worth of other data. The telnet client type uses
 *	that int as initial flag settings and negotiation options.
 */

int
socket_addClientType(NetClient_t *netClient, int port, int auxData)
{
    ClientType_t *clt;

    clt = malloc(sizeof(ClientType_t));
    if (clt == NULL) {
	return 1;
    }
    clt->clt_next = ClientTypes;
    clt->clt_netClient = netClient;
    clt->clt_port = (unsigned short int) port;
    clt->clt_auxData = auxData;
    if (! setupSocket(&clt->clt_socket, (unsigned short int) port)) {
	free(clt);
	return 1;
    }
    ClientTypes = clt;
    DEBUG(("added client type on port %d auxData 0x%x\n", port, auxData));
    return 0;
}

/*
 * socket_setBlank - outer code wants to change whether or not user input
 *	is currently being blanked (e.g. password blanking). Vector
 *	off to client-type specific code.
 */

void
socket_setBlank(GenericClient_t *gcl, int blankOn)
{

    if (gcl->gcl_type->clt_netClient->nc_setBlank != NULL) {
	(*gcl->gcl_type->clt_netClient->nc_setBlank)
	    ((void *) gcl->gcl_clientData, (bool_t) (blankOn != 0));
    }
}

/*
 * socket_setPrompt - vector off to client-type specific code to set the
 *	current prompt.
 */

void
socket_setPrompt(GenericClient_t *gcl, const char *newPrompt)
{

    if (gcl->gcl_type->clt_netClient->nc_setPrompt != NULL) {
	(*gcl->gcl_type->clt_netClient->nc_setPrompt)
	    ((void *) gcl->gcl_clientData, newPrompt);
    }
}

/*
 * socket_newName - vector off to any client-type specific code that cares
 *	about the user changing names. In AmigaMUD, a binary client
 *	used the name in any logging it did.
 */

void
socket_newName(GenericClient_t *gcl, const char *newName)
{

    if (gcl->gcl_type->clt_netClient->nc_newName != NULL) {
	(*gcl->gcl_type->clt_netClient->nc_newName)
	    ((void *) gcl->gcl_clientData, newName);
    }
}

/*
 * socket_putText - required entry point. This is called by external code
 *	to send a Request_t full of text to the indicated client. We
 *	vector off to the appropriate code. A binary client might just
 *	send it directly, but a telnet client will have to filter the
 *	text through the telnet protocol.
 */

void
socket_putText(GenericClient_t *gcl, Request_t *rq)
{

    /* There had better be one of these! */
    (*gcl->gcl_type->clt_netClient->nc_putText)
	((void *) gcl->gcl_clientData, rq);
}

/*
 * socket_flush - vector off to a client-type specific routine to flush
 *	out any buffers it may be keeping for this client. This kind
 *	of buffering can reduce the number of system calls that we do,
 *	and also reduce the number of network packets we send.
 */

void
socket_flush(GenericClient_t *gcl)
{

    if (gcl->gcl_type->clt_netClient->nc_flush != NULL) {
	(*gcl->gcl_type->clt_netClient->nc_flush)
	    ((void *) gcl->gcl_clientData);
    }
}

/*
 * Additional interfaces for DevMUD dynamic loading.
 */

struct interface socket_supplies[] = {
    { "socket_addClientType", (void *) socket_addClientType,
      "int(NetClient_t*,int,int)",
      "Add a new type of client to the server" },
    { "socket_killClient", (void *) socket_killClient,
      "void(GenericClient_t*)",
      "Force disconnection of the given client" },
    { "socket_setBlank", (void *) socket_setBlank,
      "void(GenericClient_t*,int)",
      "Control whether user input is blanked" },
    { "socket_setPrompt", (void *) socket_setPrompt,
      "void(GenericClient_t*,char*",
      "Set a new prompt for the client" },
    { "socket_newName", (void *) socket_newName,
      "void(GenericClient_t*,char*",
      "Set a new player name for the client" },
    { "socket_putText", (void *) socket_putText,
      "void(GenericClient_t*,Request_t*)",
      "Output a request of text to the user" },
    { "socket_flush", (void *) socket_flush,
      "void(GenericClient_t*)",
      "Flush any buffered output for the client" },
    { "socket_getWindowSize", (void *) socket_getWindowSize,
      "void(GenericClient_t*,int*,int*)",
      "Return the character size of the user's text display" },
    { "socket_getTerminalType", (void *) socket_getTerminalType,
      "char*(GenericClient_t*)",
      "Return the type of the user's terminal or client" },
    { "socket_addTimedAction", (void *) socket_addTimedAction,
      "int(void*,void(*)(void*,void*,int)",
      "Add a time-based event to be triggered" },
    { "socket_cancelActions", (void *) socket_cancelActions,
      "void(void*)",
      "Remove all pending actions with given world data" },
    { NULL, NULL, NULL, NULL }
};

struct interface socket_uses[] = {
    { "world_newClient", NULL, "WorldClient_t*(GenericClient_t*)", NULL },
    { "world_handleLine", NULL, "void(WorldClient_t*,char*,int)", NULL },
    { "world_clientGone", NULL, "void(WorldClient_t*)", NULL },
    { NULL, NULL, NULL, NULL }
};
