/*
 * clients.c - upper-level client handling code.
 *	- Chris Gray, cg@ami-cg.GraySage.Edmonton.AB.CA, Nov/Dec 1998.
 */

/*
 * This code is provided as-is. No warranty of any kind is given.
 */

#include <time.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <ctype.h>

#include "../include/devmud.h"
#include "../include/types.h"
#include "../include/netClient.h"
#include "server.h"

#define LINE_LENGTH	4000

typedef enum {
    cs_name,		/* want a player name */
    cs_password1,	/* want a player password */
    cs_password2,	/* first retry */
    cs_password3,	/* second and last retry */
    cs_connected,	/* fully connected */
    cs_createQuery,	/* want to create new one? */
    cs_createPW1,	/* first copy of new player password */
    cs_createPW2	/* second copy of new player password */
} ClientState_t;

typedef struct Client {
    struct Client *cl_next;
    struct Client *cl_prev;
    void *cl_genericClient;
    Player_t *cl_player;
    int cl_inputLength;
    ClientState_t cl_state;
    bool_t cl_exiting;
    bool_t cl_isMachine;
    bool_t cl_deleted;
    char cl_inputLine[LINE_LENGTH];
} Client_t;

static void (*KillClient)(void *gcl);
static void (*SetBlank)(void *gcl, int flag);
static void (*SetPrompt)(void *gcl, const char *newPrompt);
static void (*PutText)(void *gcl, Request_t *rq);
static void (*Flush)(void *gcl);
static void (*GetWindowSize)(void *gcl, int *pRows, int *pCols);
static const char *(*GetTerminalType)(void *gcl);

static int (*AddTimedAction)(int, void(*)(void *, void *, int),
			     void *cl, void *arg1, int arg2);
static void (*CancelActions)(void *cl);

static char *DbPath = "./devmud.db";
static Client_t *Clients;
static Client_t *ActiveClient;
static char UserMessage[LINE_LENGTH], PrintMessage[LINE_LENGTH];
static int ScanPos;

void
mudAbort(char *message)
{

    fprintf(stderr, "%s\n", message);
    exit(1);
}

void *
mudAlloc(unsigned int len)
{
    void *p;

    p = malloc(len);
    if (p == NULL) {
	mudAbort("out of memory");
    }
/*printf("mudAlloc(%d) => 0x%08x\n", len, p);fflush(stdout);*/
    return(p);
}

void
mudFree(void *p)
{

/*printf("mudFree(0x%x)\n", p);fflush(stdout);*/
    free(p);
}

char *
mudStrDup(const char *s)
{
    char *p;

    p = mudAlloc(strlen(s) + sizeof(char));
    strcpy(p, s);
    return(p);
}

/*
 * world_newClient - called by networking code when a new client comes in.
 */

Client_t *
world_newClient(void *gcl)
{
    Client_t *cl;

    cl = (Client_t *) mudAlloc(sizeof(Client_t));
    if (cl == NULL) {
	return NULL;
    }
    cl->cl_genericClient = gcl;
    cl->cl_state = cs_name;
    cl->cl_player = NULL;
    cl->cl_next = Clients;
    if (Clients != NULL) {
	Clients->cl_prev = cl;
    }
    cl->cl_prev = NULL;
    cl->cl_inputLength = 0;
    cl->cl_exiting = FALSE;
    cl->cl_isMachine = FALSE;
    cl->cl_deleted = FALSE;
    Clients = cl;
    cprintf(cl, "Enter player name: ");
    return cl;
}

static void
doFlush(Client_t *cl)
{
    if (Flush != NULL) {
	(*Flush)(cl->cl_genericClient);
    }
}

static void
sendToClient(Client_t *cl, char *buf, int len)
{
    Request_t *rq;

    rq = mudAlloc(sizeof(Request_t) - 2 + len);
    rq->rq_next = NULL;
    rq->rq_id = 0;
    rq->rq_availLen = len;
    rq->rq_usedLen = len;
    memcpy(&rq->rq_u.ru_text[0], buf, len);
    (*PutText)(cl->cl_genericClient, rq);
    if (cl != ActiveClient) {
	doFlush(cl);
    }
}

void
aprintf(Client_t *active, const char *fmt, ...)
{
    va_list ap;
    int len;
    Client_t *cl;
    Player_t *pl;
    Thing_t *here;

    va_start(ap, fmt);
    vsprintf(&PrintMessage[0], fmt, ap);
    va_end(ap);
    len = strlen(&PrintMessage[0]);
    if (len >= LINE_LENGTH) {
	mudAbort("message too long in aprintf");
    }
    if (active == NULL || active->cl_player == NULL) {
	here = NULL;
    } else {
	here = active->cl_player->pl_location;
    }
    cl = Clients;
    while (cl != NULL) {
	pl = cl->cl_player;
	if (! cl->cl_deleted && pl != NULL && cl != active &&
	    (active == NULL || pl->pl_location == here))
	{
	    if (cl->cl_isMachine) {
		runWithString(cl, pl, &PrintMessage[0], "LISTEN_ACTION");
	    } else {
		sendToClient(cl, &PrintMessage[0], len);
	    }
	}
	cl = cl->cl_next;
    }
}

void
va_cprintf(Client_t *cl, const char *fmt, va_list ap)
{
    int len;

    vsprintf(&PrintMessage[0], fmt, ap);
    len = strlen(&PrintMessage[0]);
    if (len >= LINE_LENGTH) {
	mudAbort("message too long in cprintf");
    }
    if (! cl->cl_isMachine) {
	sendToClient(cl, &PrintMessage[0], len);
#if MACHINE_DEBUG
    } else {
	(void) write(1, &PrintMessage[0], len);
#endif
    }
}

void
cprintf(Client_t *cl, const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    va_cprintf(cl, fmt, ap);
    va_end(ap);
}

Client_t *
findClient(Thing_t *who)
{
    Client_t *cl;

    cl = Clients;
    while (cl != NULL) {
	if (! cl->cl_deleted && cl->cl_player != NULL &&
	    &cl->cl_player->pl_thing == who)
	{
	    return(cl);
	}
	cl = cl->cl_next;
    }
    return(NULL);
}

Thing_t *
findActivePlayer(char *name, Thing_t *where)
{
    Client_t *cl;

    cl = Clients;
    while (cl != NULL) {
	if (! cl->cl_deleted && cl->cl_player != NULL &&
	    strcmp(cl->cl_player->pl_thing.th_name, name) == 0 &&
	    (where == NULL || cl->cl_player->pl_location == where))
	{
	    return(&cl->cl_player->pl_thing);
	}
	cl = cl->cl_next;
    }
    return(NULL);
}

int
forEachPlayer(char *proc)
{
    Client_t *cl;
    int count;

    count = 0;
    cl = Clients;
    while (cl != NULL) {
	if (! cl->cl_deleted && cl->cl_player != NULL &&
	    cl->cl_state == cs_connected)
	{
	    runExplicitWithPlayer(cl, cl->cl_player, proc);
	    count += 1;
	}
	cl = cl->cl_next;
    }
    return(count);
}

Thing_t *
clientLocation(Thing_t *who)
{
    Client_t *cl;
    Player_t *pl;

    cl = Clients;
    while (cl != NULL) {
	if (! cl->cl_deleted) {
	    pl = cl->cl_player;
	    if (pl != NULL && &pl->pl_thing == who) {
		return(pl->pl_location);
	    }
	}
	cl = cl->cl_next;
    }
    return(NULL);
}

void
setClientLocation(Thing_t *who, Thing_t *location)
{
    Client_t *cl;
    Player_t *pl;

    cl = Clients;
    while (cl != NULL) {
	if (! cl->cl_deleted) {
	    pl = cl->cl_player;
	    if (pl != NULL && &pl->pl_thing == who) {
		pl->pl_location = location;
		return;
	    }
	}
	cl = cl->cl_next;
    }
}

void
setQuit(Client_t *cl)
{
    cl->cl_exiting = TRUE;
}

void
initMachine(Player_t *pl)
{
    Client_t *cl;

    cl = (Client_t *) mudAlloc(sizeof(Client_t));
    cl->cl_prev = NULL;
    cl->cl_next = Clients;
    cl->cl_player = pl;
    cl->cl_state = cs_connected;
    cl->cl_exiting = FALSE;
    cl->cl_isMachine = TRUE;
    cl->cl_deleted = FALSE;
    cl->cl_prev = NULL;
    cl->cl_next = Clients;
    Clients = cl;
}

Thing_t *
createMachine(char *name)
{
    Player_t *pl;

    pl = newMachine(name);
    pl->pl_password = mudStrDup("XXX");
    initMachine(pl);
    return(&pl->pl_thing);
}

static void
unlinkClient(Client_t *cl)
{

    if (cl->cl_prev != NULL) {
	cl->cl_prev->cl_next = cl->cl_next;
    } else {
	Clients = cl->cl_next;
    }
    if (cl->cl_next != NULL) {
	cl->cl_next->cl_prev = cl->cl_prev;
    }
    mudFree(cl);
}

bool_t
destroyMachine(Thing_t *th)
{
    Client_t *cl;

    cl = Clients;
    while (cl != NULL &&
	(cl->cl_player == NULL || &cl->cl_player->pl_thing != th))
    {
	cl = cl->cl_next;
    }
    if (cl != NULL) {
	(*CancelActions)(cl);
	cl->cl_deleted = TRUE;
	return(TRUE);
    } else {
	return(FALSE);
    }
}

static void
doAfterAction(void *worldData, void *action, int dummy)
{
    Client_t *cl;

    cl = (Client_t *) worldData;
    runExplicitWithPlayer(cl, cl->cl_player, (char *) action);
}

void
doAfter(Client_t *cl, int delay, char *proc)
{

    (void) (*AddTimedAction)(delay, doAfterAction, cl, proc, 0);
}

bool_t
doForce(Thing_t *who, char *proc)
{
    Client_t *cl;
    Player_t *pl;

    cl = Clients;
    while (cl != NULL) {
	if (! cl->cl_deleted) {
	    pl = cl->cl_player;
	    if (pl != NULL && &pl->pl_thing == who) {
		runExplicitWithPlayer(cl, pl, proc);
		return(TRUE);
	    }
	}
	cl = cl->cl_next;
    }
    return(FALSE);
}

static void
startMachines(void)
{
    Client_t *cl;

    cl = Clients;
    while (cl != NULL) {
	if (cl->cl_isMachine) {
	    runWithPlayer(cl, cl->cl_player, "MACHINE_RESTART_ACTION");
	}
	cl = cl->cl_next;
    }
}

static void
cleanupMachines(void)
{
    Client_t **pcl, *cl;

    pcl = &Clients;
    while ((cl = *pcl) != NULL) {
	if (cl->cl_deleted) {
	    if (cl->cl_player != NULL) {
		deleteMachine(cl->cl_player);
	    }
	    unlinkClient(cl);
	} else {
	    pcl = &cl->cl_next;
	}
    }
}

static void
killPlayer(Client_t *cl)
{

    (*KillClient)(cl->cl_genericClient);
    if (cl->cl_player != NULL) {
	aprintf(cl, "%s has left the game.\n",cl->cl_player->pl_thing.th_name);
    } else {
	aprintf(cl, "??? has left the game.\n");
    }
    (*CancelActions)(cl);
    unlinkClient(cl);
}

static void
setBlank(Client_t *cl, bool_t flag)
{

    if (SetBlank != NULL) {
	(*SetBlank)(cl->cl_genericClient, flag);
    }
}

static void
enterStuff(Client_t *cl, bool_t isNewPlayer)
{
    Client_t *cl2;
    const char *ttyType;
    int nRows, nCols;
    bool_t first, gotSize, gotEither;

    setBlank(cl, FALSE);
    cprintf(cl, "Welcome to ToyMUD V1.0!\n");
    cl2 = Clients;
    first = TRUE;
    while (cl2 != NULL) {
	if (! cl2->cl_deleted && cl2 != cl && ! cl2->cl_isMachine) {
	    if (first) {
		first = FALSE;
		cprintf(cl, "Currently connected: ");
	    }
	    if (cl2->cl_player == NULL) {
		cprintf(cl, "??? ");
	    } else {
		cprintf(cl, "%s", cl2->cl_player->pl_thing.th_name);
		gotSize = FALSE;
		gotEither = FALSE;
		if (GetWindowSize != NULL) {
		    GetWindowSize(cl2->cl_genericClient, &nRows, &nCols);
		    if (nRows != 0 && nCols != 0) {
			cprintf(cl, "[%dx%d", nRows, nCols);
			gotSize = TRUE;
			gotEither = TRUE;
		    }
		}
		if (GetTerminalType != NULL) {
		    ttyType = (*GetTerminalType)(cl2->cl_genericClient);
		    if (ttyType != NULL) {
			if (gotSize) {
			    cprintf(cl, " ");
			} else {
			    cprintf(cl, "[");
			}
			cprintf(cl, "%s", ttyType);
			gotEither = TRUE;
		    }
		}
		if (gotEither) {
		    cprintf(cl, "] ");
		}
	    }
	}
	cl2 = cl2->cl_next;
    }
    if (first) {
	cprintf(cl, "No-one else is currently connected.\n\n");
    } else {
	cprintf(cl, "\n\n");
    }
    cl->cl_state = cs_connected;
    if (isNewPlayer) {
	runWithPlayer(cl, cl->cl_player, "NEW_PLAYER_ACTION");
    } else {
	runWithPlayer(cl, cl->cl_player, "RESTART_ACTION");
    }
    doFlush(cl);
    if (SetPrompt != NULL) {
	(*SetPrompt)(cl->cl_genericClient, "input> ");
    }
    aprintf(cl, "%s has entered the game.\n",
	    cl->cl_player->pl_thing.th_name);
}

static void
skipBlanks(void)
{
    char *p;

    p = &UserMessage[ScanPos];
    while (isspace(*p)) {
	++p;
	ScanPos += 1;
    }
}

char *
getWord(void)
{
    char *p, *q;
    char save;

    skipBlanks();
    p = &UserMessage[ScanPos];
    q = p;
    while (*p != '\0' && ! isspace(*p)) {
	++p;
	ScanPos += 1;
    }
    if (*p != '\0') {
	save = *p;
	*p = '\0';
	ScanPos += 1;
	q = mudStrDup(q);
	*p = save;
    } else {
	q = mudStrDup(q);
    }
    return(q);
}

char *
getTail(void)
{
    skipBlanks();
    return(mudStrDup(&UserMessage[ScanPos]));
}

void
setTail(char *st)
{
    if ((int) strlen(st) >= LINE_LENGTH) {
	mudAbort("line too long in setTail");
    }
    strcpy(&UserMessage[0], st);
    ScanPos = 0;
}

void
world_clientGone(Client_t *cl)
{
    if (cl->cl_state == cs_connected) {
	runWithPlayer(cl, cl->cl_player, "QUIT_ACTION");
	doFlush(cl);
    }
    killPlayer(cl);
}

/*
 * world_handleLine - this is the routine that is called to handle an
 *	input line from the remote client. We get a client pointer,
 *	a buffer pointer, and a buffer length. The length will not
 *	include the terminating '\0'.
 */

void
world_handleLine(Client_t *cl, char *buffer, unsigned int len)
{
    int totalLen;
    char *w;
    Player_t *pl;
    bool_t noProcess;

    if (len == 0) {
	return;
    }
    memcpy(&UserMessage[0], buffer, len + 1);
    noProcess = FALSE;
    if (UserMessage[len - 1] == '-') {
	UserMessage[len - 1] = '\n';
	noProcess = TRUE;
    } else if (UserMessage[len - 1] == '+') {
	len -= 1;
	noProcess = TRUE;
    }
    totalLen = cl->cl_inputLength;
    if (totalLen + len > LINE_LENGTH) {
	len = LINE_LENGTH - totalLen;
    }
    if (len != 0) {
	memcpy(&cl->cl_inputLine[totalLen], &UserMessage[0], len);
	totalLen += len;
    }
    if (noProcess) {
	cl->cl_inputLength = totalLen;
	return;
    }
    memcpy(&UserMessage[0], &cl->cl_inputLine[0], totalLen);
    UserMessage[totalLen] = '\0';
    cl->cl_inputLength = 0;

    ScanPos = 0;
    skipBlanks();
    if (UserMessage[ScanPos] == '\0') {
	if (cl->cl_state != cs_connected) {
	    killPlayer(cl);
	}
	return;
    }

    ActiveClient = cl;
    switch (cl->cl_state) {
    case cs_name:
	w = getWord();
	pl = findPlayer(w);
	if (pl == NULL) {
	    cprintf(cl,
	      "Player '%s' does not exist. Do you wish to create it? ", w);
	    cl->cl_player = newPlayer(mudStrDup(w));
	    cl->cl_state = cs_createQuery;
	} else {
	    cprintf(cl, "Enter player password: ");
	    setBlank(cl, TRUE);
	    cl->cl_player = pl;
	    cl->cl_state = cs_password1;
	}
	break;
    case cs_password1:
	w = getWord();
	if (strcmp(w, cl->cl_player->pl_password) == 0) {
	    enterStuff(cl, FALSE);
	} else {
	    cprintf(cl, "Password incorrect, try again: ");
	    cl->cl_state = cs_password2;
	}
	break;
    case cs_password2:
	w = getWord();
	if (strcmp(w, cl->cl_player->pl_password) == 0) {
	    enterStuff(cl, FALSE);
	} else {
	    cprintf(cl, "Password incorrect, try again: ");
	    cl->cl_state = cs_password3;
	}
	break;
    case cs_password3:
	w = getWord();
	if (strcmp(w, cl->cl_player->pl_password) == 0) {
	    enterStuff(cl, FALSE);
	} else {
	    cprintf(cl, "Password incorrect.\n");
	    killPlayer(cl);
	}
	break;
    case cs_connected:
	if (len > 0) {
	    if (UserMessage[ScanPos] == '.') {
		ScanPos += 1;
		skipBlanks();
		runProc(cl, cl->cl_player, &UserMessage[ScanPos]);
	    } else {
		strcpy(&UserMessage[0], &UserMessage[ScanPos]);
		ScanPos = 0;
		runWithString(cl, cl->cl_player, &UserMessage[0],
			      "PARSE_ACTION");
	    }
	    doFlush(cl);
	}
	if (cl->cl_exiting) {
	    runWithPlayer(cl, cl->cl_player, "QUIT_ACTION");
	    doFlush(cl);
	    killPlayer(cl);
	}
	break;
    case cs_createQuery:
	w = getWord();
	if (*w == 'y' || *w == 'Y') {
	    cprintf(cl, "Enter new player password: ");
	    cl->cl_state = cs_createPW1;
	    setBlank(cl, TRUE);
	} else {
	    deletePlayer(cl->cl_player);
	    killPlayer(cl);
	}
	break;
    case cs_createPW1:
	w = getWord();
	cl->cl_player->pl_password = mudStrDup(w);
	cprintf(cl, "Re-enter password for verification: ");
	cl->cl_state = cs_createPW2;
	break;
    case cs_createPW2:
	w = getWord();
	if (strcmp(cl->cl_player->pl_password, w) == 0) {
	    enterStuff(cl, TRUE);
	} else {
	    cprintf(cl, "Password not verified.\n");
	    deletePlayer(cl->cl_player);
	    killPlayer(cl);
	}
	break;
    }
    ActiveClient = NULL;
}

#if 0
		if (FD_ISSET(cl->cl_fd, &exceptFds)) {
		    if (cl->cl_state == cs_createQuery ||
			cl->cl_state == cs_createPW1 ||
			cl->cl_state == cs_createPW2)
		    {
			deletePlayer(cl->cl_player);
			cl->cl_player = NULL;
		    } else {
			runWithPlayer(cl, cl->cl_player, "QUIT_ACTION");
			doFlush(cl);
		    }
		    killPlayer(cl);
#endif

int
world_initialize(void)
{

    KillClient = NULL;
    SetBlank = NULL;
    SetPrompt = NULL;
    PutText = NULL;
    Flush = NULL;
    GetWindowSize = NULL;
    GetTerminalType = NULL;
    AddTimedAction = NULL;
    CancelActions = NULL;
    return 0;
}

void
world_use_functions(struct interface *it)
{

    while (it->name != NULL) {
	if (strcmp(it->name, "socket_killClient") == 0) {
	    KillClient = (void (*)(void *)) it->function;
	}
	if (strcmp(it->name, "socket_setBlank") == 0) {
	    SetBlank = (void (*)(void *, int)) it->function;
	}
	if (strcmp(it->name, "socket_setPrompt") == 0) {
	    SetPrompt = (void (*)(void *, const char *)) it->function;
	}
	/* We have no way to change the player name, so we cannot use
	   socket_newName */
	if (strcmp(it->name, "socket_putText") == 0) {
	    PutText = (void (*)(void *, Request_t *)) it->function;
	}
	if (strcmp(it->name, "socket_flush") == 0) {
	    Flush = (void (*)(void *)) it->function;
	}
	if (strcmp(it->name, "socket_getWindowSize") == 0) {
	    GetWindowSize = (void (*)(void *, int *, int*)) it->function;
	}
	if (strcmp(it->name, "socket_getTerminalType") == 0) {
	    GetTerminalType = (const char *(*)(void *)) it->function;
	}

	if (strcmp(it->name, "socket_addTimedAction") == 0) {
	    AddTimedAction = (int (*)(int, void (*)(void *, void *, int),
				      void *, void *, int)) it->function;
	}
	if (strcmp(it->name, "socket_cancelActions") == 0) {
	    CancelActions = (void (*)(void *)) it->function;
	}
	++it;
    }
}

int
world_start(void)
{
    Variable_t *globalContext;

    /* Only check for the ones we *must* have */
    if (PutText == NULL || KillClient == NULL ||
	AddTimedAction == NULL || CancelActions == NULL)
    {
	return 1;
    }
    Clients = NULL;
    ActiveClient = NULL;
    globalContext = readDB(DbPath);
    initInterpreter(globalContext);
    srand(time(NULL));
    startMachines();
    return 0;
}

void
world_stop(void)
{

    writeDB(DbPath, getGlobals());
    cleanupMachines();
}

struct interface world_supplies[] = {
    { "world_newClient", (void *) world_newClient,
      "WorldClient_t*(GenericClient_t*)",
      "Handle the connection of a new client" },
    { "world_handleLine", (void *) world_handleLine,
      "void(WorldClient_t*,char*,int)",
      "Handle an input line from the user" },
    { "world_clientGone", (void *) world_clientGone,
      "void(WorldClient_t*)",
      "Handle the disconnection of a client" },
    { NULL, NULL, NULL, NULL }
};

struct interface world_uses[] = {
    { "socket_killClient", NULL, "void(GenericClient_t*)", NULL },
    { "socket_setBlank", NULL, "void(GenericClient_t*,int)", NULL },
    { "socket_setPrompt", NULL, "void(GenericClient_t*,char*", NULL },
    { "socket_newName", NULL, "void(GenericClient_t*,char*", NULL },
    { "socket_putText", NULL, "void(GenericClient_t*,Request_t*)", NULL },
    { "socket_flush", NULL, "void(GenericClient_t*)", NULL },
    { "socket_getWindowSize", NULL, "void(GenericClient_t*,int*,int*)", NULL },
    { "socket_getTerminalType", NULL, "char*(GenericClient_t*)", NULL },
    { "socket_addTimedAction",NULL,"int(void*,void(*)(void*,void*,int)",NULL },
    { "socket_cancelActions", NULL, "void(void*)", NULL },
    { NULL, NULL, NULL, NULL }
};
