#include <stdio.h>
#include <ctype.h>
#include <stdarg.h>
#include <string.h>

#include "../include/types.h"
#include "server.h"

#define BUFF_LEN    100
#define PROC_LEN    10000

static char Buffer[BUFF_LEN];
static char ProcBuffer[PROC_LEN];
static FILE *Fd;

static Thing_t *Things;
static Thing_t **ThingTail;
static int ThingNext;

static Player_t *Players, *PlayerTail;
static Player_t *Machines, *MachineTail;

static bool_t
getLine(void)
{

    do {
	if (fgets(Buffer, BUFF_LEN, Fd) == NULL) {
	    return(FALSE);
	}
    } while (Buffer[0] == '#' || Buffer[0] == '\n');
    Buffer[strlen(&Buffer[0]) - 1] = '\0';
    return(TRUE);
}

Thing_t *
findThing(char *name)
{
    Thing_t *th;

    th = Things;
    while (th != NULL) {
	if (strcmp(th->th_name, name) == 0) {
	    return(th);
	}
	th = th->th_next;
    }
    return(th);
}

Thing_t *
newThing(char *name)
{
    Thing_t *th;
    char buf[50];

    th = (Thing_t *) mudAlloc(sizeof(Thing_t));
    th->th_next = NULL;
    if (*name == '\0') {
	sprintf(&buf[0], "t%d", ThingNext);
	ThingNext += 1;
	name = mudStrDup(&buf[0]);
    }
    th->th_name = name;
    th->th_context = NULL;
    th->th_written = FALSE;
    *ThingTail = th;
    ThingTail = &th->th_next;
    return(th);
}

bool_t
destroyThing(Thing_t *theThing)
{
    Thing_t *th;
    Thing_t **pTh;

    pTh = &Things;
    while ((th = *pTh) != NULL && th != theThing) {
	pTh = &th->th_next;
    }
    if (th == NULL) {
	return(FALSE);
    }
    if (ThingTail == &th->th_next) {
	ThingTail = pTh;
    }
    *pTh = th->th_next;
    mudFree(th->th_name);
    mudFree(th);
    return(TRUE);
}

static Player_t *
newEntity(Player_t **pPlayers, Player_t **pPlayerTail, char *name)
{
    Player_t *pl;

    pl = (Player_t *) mudAlloc(sizeof(Player_t));
    pl->pl_next = NULL;
    pl->pl_thing.th_name = name;
    pl->pl_password = NULL;
    pl->pl_location = NULL;
    pl->pl_thing.th_context = NULL;
    if (*pPlayers == NULL) {
	*pPlayers = pl;
	pl->pl_prev = NULL;
    } else {
	pl->pl_prev = PlayerTail;
	(*pPlayerTail)->pl_next = pl;
    }
    *pPlayerTail = pl;
    return(pl);
}

static void
deleteEntity(Player_t **pPlayers, Player_t **pPlayerTail, Player_t *pl)
{

    if (pl->pl_prev != NULL) {
	pl->pl_prev->pl_next = pl->pl_next;
    } else {
	*pPlayers = pl->pl_next;
    }
    if (pl->pl_next != NULL) {
	pl->pl_next->pl_prev = pl->pl_prev;
    } else {
	*pPlayerTail = pl->pl_prev;
    }
    if (pl->pl_thing.th_name != NULL) {
	mudFree(pl->pl_thing.th_name);
    }
    mudFree(pl);
}

Player_t *
newPlayer(char *name)
{
    return(newEntity(&Players, &PlayerTail, name));
}

void
deletePlayer(Player_t *pl)
{
    if (pl->pl_password != NULL) {
	mudFree(pl->pl_password);
    }
    deleteEntity(&Players, &PlayerTail, pl);
}

Player_t *
findPlayer(char *name)
{
    Player_t *pl;;

    pl = Players;
    while (pl != NULL) {
	if (strcmp(pl->pl_thing.th_name, name) == 0) {
	    return(pl);
	}
	pl = pl->pl_next;
    }
    return(pl);
}

Player_t *
newMachine(char *name)
{
    return(newEntity(&Machines, &MachineTail, name));
}

void
deleteMachine(Player_t *pl)
{
    if (pl->pl_password != NULL) {
	mudFree(pl->pl_password);
    }
    deleteEntity(&Machines, &MachineTail, pl);
}

static Variable_t *
readContext(Thing_t *theThing)
{
    Variable_t *head, *v;
    Variable_t **tail;
    Thing_t *th;
    char *p, *q, *name, *name2, *thingName;
    int n;
    bool_t negative, wasPair, wasDummy;

    /* hush up gcc */
    v = NULL;
    name2 = NULL;
    if (theThing == NULL) {
	thingName = "GLOBAL";
    } else {
	thingName = theThing->th_name;
    }
    head = NULL;
    tail = &head;
    while (getLine() && Buffer[0] != '.') {
	p = &Buffer[0];
	q = p;
	while (*p != '\0' && ! isspace(*p) && *p != '/') {
	    ++p;
	}
	wasPair = *p == '/';
	*p++ = '\0';
	name = mudStrDup(q);
	if (strcmp(name, "-") == 0) {
	    wasDummy = 1;
	} else {
	    wasDummy = 0;
	    v = head;
	    while (v != NULL) {
		if (strcmp(v->v_name, name) == 0) {
		    fprintf(stderr, "%s/%s: duplicate property in context\n",
			    thingName, name);
		    mudAbort(NULL);
		}
		v = v->v_next;
	    }
	    v = (Variable_t *) mudAlloc(sizeof(Variable_t));
	    v->v_name = name;
	    v->v_next = NULL;
	    *tail = v;
	    tail = &v->v_next;
	}
	if (wasPair) {
	    q = p;
	    while (*p != '\0' && ! isspace(*p)) {
		++p;
	    }
	    *p++ = '\0';
	    name2 = mudStrDup(q);
	}
	while (*p != '\0' && isspace(*p)) {
	    ++p;
	}
	if (*p == '\0') {
	    if (! getLine()) {
		fprintf(stderr, "%s/%s: missing property value\n",
			thingName, name);
		mudAbort(NULL);
	    }
	    p = &Buffer[0];
	}
	if (isdigit(*p) || *p == '-') {
	    if (wasDummy) {
		fprintf(stderr, "%s/%s: can't use dummy name with integer\n",
			thingName, name);
		mudAbort(NULL);
	    }
	    v->v_value.v_type = t_int;
	    if (*p == '-') {
		++p;
		negative = TRUE;
	    } else {
		negative = FALSE;
	    }
	    n = 0;
	    while (isdigit(*p)) {
		n *= 10;
		n += (*p - '0');
		++p;
	    }
	    v->v_value.v_.v_int = negative ? -n : n;
	} else if (*p == '`') {
	    if (wasDummy) {
		fprintf(stderr, "%s/%s: can't use dummy name with proc\n",
			thingName, name);
		mudAbort(NULL);
	    }
	    v->v_value.v_type = t_proc;
	    ++p;
	    q = &ProcBuffer[0];
	    n = 0;
	    while (*p != '`') {
		if (n >= PROC_LEN) {
		    fprintf(stderr, "%s/%s: proc too long\n",
			    thingName, name);
		    mudAbort(NULL);
		}
		if (*p == '\0') {
		    if (! getLine()) {
			fprintf(stderr, "%s/%s: unterminated proc\n",
				thingName, name);
			mudAbort(NULL);
		    }
		    p = &Buffer[0];
		    *q++ = '\n';
		} else {
		    *q++ = *p++;
		}
	    }
	    *q = '\0';
	    v->v_value.v_.v_proc = mudStrDup(&ProcBuffer[0]);
	} else if (*p == '"') {
	    if (wasDummy) {
		fprintf(stderr, "%s/%s: can't use dummy name with string\n",
			thingName, name);
		mudAbort(NULL);
	    }
	    v->v_value.v_type = t_string;
	    ++p;
	    q = &ProcBuffer[0];
	    n = 0;
	    while (*p != '\"') {
		if (n >= PROC_LEN) {
		    fprintf(stderr, "%s/%s: string too long\n",
			    thingName, name);
		    mudAbort(NULL);
		}
		if (*p == '\0') {
		    if (! getLine()) {
			fprintf(stderr, "%s/%s: unterminated string\n",
				thingName, name);
			mudAbort(NULL);
		    }
		    p = &Buffer[0];
		    *q++ = '\n';
		} else if (*p == '\\') {
		    ++p;
		    switch (*p) {
		    case 'n':
			*q++ = '\n';
			++p;
			break;
		    case '"':
			*q++ = '"';
			++p;
			break;
		    case '\0':
			break;
		    default:
			*q++ = *p++;
			break;
		    }
		} else {
		    *q++ = *p++;
		}
	    }
	    *q = '\0';
	    v->v_value.v_.v_string = mudStrDup(&ProcBuffer[0]);
	} else if (isalpha(*p)) {
	    q = p;
	    while (isalnum(*p) || *p == '_') {
		++p;
	    }
	    *p = '\0';
	    if (strcmp(q, "nil") == 0) {
		th = NULL;
	    } else {
		th = findThing(q);
		if (th == NULL) {
		    fprintf(stderr, "%s/%s: thing '%s' does not exist\n",
			    thingName, name, q);
		    mudAbort(NULL);
		}
	    }
	    if (! wasDummy) {
		v->v_value.v_type = t_thing;
		v->v_value.v_.v_thing = th;
	    }
	    if (wasPair && theThing != NULL) {
		if (th == NULL) {
		    fprintf(stderr, "%s/%s: can't have reverse from nil\n",
			    thingName, name);
		    mudAbort(NULL);
		}
		v = th->th_context;
		while (v != NULL) {
		    if (strcmp(v->v_name, name2) == 0) {
			fprintf(stderr,
				"%s: %s/%s: duplicate property in context\n",
				thingName, th->th_name, name2);
			mudAbort(NULL);
		    }
		    v = v->v_next;
		}
		v = (Variable_t *) mudAlloc(sizeof(Variable_t));
		v->v_next = th->th_context;
		v->v_name = name2;
		v->v_value.v_type = t_thing;
		v->v_value.v_.v_thing = theThing;
		th->th_context = v;
	    }
	} else {
	    fprintf(stderr, "%s/%s: illegal character '%c' for value'\n",
		    thingName, name, *p);
	    mudAbort(NULL);
	}
    }
    return(head);
}

static void
readEntities(char *kind, Player_t *((*findProc)(char *name)),
	     Player_t *((*newProc)(char *name)),
	     void (*initProc)(Player_t *pl))
{
    Player_t *pl;
    Thing_t *th;
    char *name;

    while (1) {
	if (! getLine()) {
	    break;
	}
	if (Buffer[0] == '\0') {
	    continue;
	}
	if (strcmp(&Buffer[0], "..") == 0) {
	    break;
	}
	name = mudStrDup(&Buffer[0]);
	if (findProc != NULL) {
	    pl = (*findProc)(name);
	    if (pl != NULL) {
		fprintf(stderr, "%s: duplicate %s name\n", name, kind);
		mudAbort(NULL);
	    }
	}
	pl = (*newProc)(name);
	if (! getLine()) {
	    fprintf(stderr, "%s: missing %s password\n", name, kind);
	    mudAbort(NULL);
	}
	pl->pl_password = mudStrDup(&Buffer[0]);
	if (! getLine()) {
	    fprintf(stderr, "%s: missing %s location\n", name, kind);
	    mudAbort(NULL);
	}
	if (strcmp(&Buffer[0], "nil") == 0) {
	    pl->pl_location = NULL;
	} else {
	    th = findThing(&Buffer[0]);
	    if (th == NULL) {
		fprintf(stderr, "%s: thing %s (location) not found\n",
			name, &Buffer[0]);
		mudAbort(NULL);
	    }
	    pl->pl_location = th;
	}
	pl->pl_thing.th_context = readContext(&pl->pl_thing);
	if (initProc != NULL) {
	    (*initProc)(pl);
	}
    }
}

Variable_t *
readDB(char *path)
{
    Thing_t *th;
    Variable_t *globalContext;
    char *name;

    Fd = fopen(path, "r");
    if (Fd == NULL) {
	fprintf(stderr, "server can't open database file '%s' for read\n",
		path);
	mudAbort(NULL);
    }

    if (! getLine()) {
	fprintf(stderr, "database file '%s' is empty\n", path);
	mudAbort(NULL);
    }
    name = &Buffer[0];
    ThingNext = 0;
    while (isdigit(*name)) {
	ThingNext *= 10;
	ThingNext += (*name - '0');
	++name;
    }
    if (*name != '\0') {
	fprintf(stderr, "invalid thing count in database file '%s'\n", path);
	mudAbort(NULL);
    }

    Things = NULL;
    ThingTail = &Things;
    while (1) {
	if (! getLine()) {
	    break;
	}
	if (Buffer[0] == '\0') {
	    continue;
	}
	if (strcmp(&Buffer[0], "..") == 0) {
	    break;
	}
	name = mudStrDup(&Buffer[0]);
	th = findThing(name);
	if (th != NULL) {
	    fprintf(stderr, "%s: duplicate thing name\n", name);
	    mudAbort(NULL);
	}
	th = newThing(name);
	th->th_context = readContext(th);
    }
    if (Things == NULL) {
	fprintf(stderr, "No things in database!\n");
	mudAbort(NULL);
    }

    globalContext = readContext(NULL);

    Players = NULL;
    PlayerTail = NULL;
    readEntities("player", findPlayer, newPlayer, NULL);
    Machines = NULL;
    MachineTail = NULL;
    readEntities("machine", NULL, newMachine, initMachine);

    fclose(Fd);
    return(globalContext);
}

static bool_t
canWriteThing(Thing_t *th)
{
    Variable_t *v, *v2;
    Thing_t *th2;
    bool_t seenMe;

    v = th->th_context;
    while (v != NULL) {
	if (v->v_value.v_type == t_thing) {
	    th2 = v->v_value.v_.v_thing;
	    if (th2 != NULL && ! th2->th_written) {
		v2 = th2->th_context;
		seenMe = FALSE;
		while (v2 != NULL) {
		    if (v2->v_value.v_type == t_thing &&
			v2->v_value.v_.v_thing == th)
		    {
			seenMe = TRUE;
			v2 = NULL;
		    } else {
			v2 = v2->v_next;
		    }
		}
		if (! seenMe) {
		    return(FALSE);
		}
	    }
	}
	v = v->v_next;
    }
    return(TRUE);
}

static void
writeContext(Variable_t *v, Thing_t *th)
{
    Variable_t *v2, *v3;
    Thing_t *th2;
    char *p;

    while (v != NULL) {
	if (v->v_value.v_type == t_thing) {
	    th2 = v->v_value.v_.v_thing;
	    if (th2 == NULL) {
		fputs(v->v_name, Fd);
		fputs(" nil\n", Fd);
	    } else {
		if (th2->th_written) {
		    if (th == NULL) {
			v2 = NULL;
		    } else {
			v2 = th2->th_context;
			while (v2 != NULL &&
			       (v2->v_value.v_type != t_thing ||
				v2->v_value.v_.v_thing != th))
			{
			    v2 = v2->v_next;
			}
		    }
		    fputs(v->v_name, Fd);
		    if (v2 != NULL) {
			fputc('/', Fd);
			fputs(v2->v_name, Fd);
			/* needed to handle rooms with double links */
			v2->v_value.v_type = t_void;
		    }
		    fputc(' ', Fd);
		    fputs(th2->th_name, Fd);
		    fputc('\n', Fd);
		    if (v2 != NULL) {
			/* needed to handle some -/name possibilities */
			v3 = v->v_next;
			while (v3 != NULL &&
			       (v3->v_value.v_type != t_thing ||
				v3->v_value.v_.v_thing != th2))
			{
			    v3 = v3->v_next;
			}
			if (v3 == NULL) {
			    v2 = v2->v_next;
			    while (v2 != NULL) {
				if (v2->v_value.v_type == t_thing &&
				    v2->v_value.v_.v_thing == th)
				{
				    fputs("-/", Fd);
				    fputs(v2->v_name, Fd);
				    fputc(' ', Fd);
				    fputs(th2->th_name, Fd);
				    fputc('\n', Fd);
				}
				v2 = v2->v_next;
			    }
			}
		    }
		}
	    }
	} else {
	    fputs(v->v_name, Fd);
	    fputc(' ', Fd);
	    switch (v->v_value.v_type) {
	    case t_int:
		fprintf(Fd, "%d", v->v_value.v_.v_int);
		break;
	    case t_string:
		p = v->v_value.v_.v_string;
		if ((int) strlen(p) >= 70) {
		    fputc('\n', Fd);
		}
		fputc('"', Fd);
		while (*p != '\0') {
		    switch (*p) {
		    case '"':
			fputs("\\\"", Fd);
			break;
		    case '\\':
			fputs("\\\\", Fd);
			break;
		    default:
			fputc(*p, Fd);
			break;
		    }
		    ++p;
		}
		fputc('"', Fd);
		break;
	    case t_proc:
		fputc('`', Fd);
		fputs(v->v_value.v_.v_proc, Fd);
		fputc('`', Fd);
		break;
	    default:
		mudAbort("writeContext - unexpected type");
	    }
	    fputc('\n', Fd);
	}
	v = v->v_next;
    }
    fputs(".\n", Fd);
}

static void
writeEntities(Player_t *pl)
{

    while (pl != NULL) {
	fputs(pl->pl_thing.th_name, Fd);
	fputc('\n', Fd);
	fputs(pl->pl_password, Fd);
	fputc('\n', Fd);
	if (pl->pl_location == NULL) {
	    fputs("nil", Fd);
	} else {
	    fputs(pl->pl_location->th_name, Fd);
	}
	fputc('\n', Fd);
	writeContext(pl->pl_thing.th_context, NULL);
	pl = pl->pl_next;
    }
    fputs("..\n", Fd);
}

void
writeDB(char *path, Variable_t *globalContext)
{
    Thing_t *th;
    bool_t doneSome, someLeft;

    Fd = fopen(path, "w");
    if (Fd == NULL) {
	fprintf(stderr, "server can't open database file '%s' for write\n",
		path);
	mudAbort(NULL);
    }

    fprintf(Fd, "%d\n", ThingNext);

    do {
	doneSome = FALSE;
	someLeft = FALSE;
	th = Things;
	while (th != NULL) {
	    if (! th->th_written) {
		if (canWriteThing(th)) {
		    fputs(th->th_name, Fd);
		    fputc('\n', Fd);
		    writeContext(th->th_context, th);
		    th->th_written = TRUE;
		    doneSome = TRUE;
		} else {
		    someLeft = TRUE;
		}
	    }
	    th = th->th_next;
	}
    } while (doneSome && someLeft);
    fputs("..\n", Fd);

    writeContext(globalContext, NULL);

    writeEntities(Players);
    writeEntities(Machines);

    fclose(Fd);
}
