#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

#include "../include/types.h"
#include "server.h"

#define STRING_MAX	1000

#define BUILT_IN_PROTO	\
    Value_t *par1, Value_t *par2, Value_t *par3, Value_t *result

typedef enum {
    tk_error, tk_eof,
    tk_number, tk_stringConstant, tk_procConstant, tk_id,
    tk_lParen, tk_rParen, tk_lBrace, tk_rBrace,
    tk_semi, tk_comma, tk_colon, tk_dot,
    tk_assign, tk_plus, tk_minus, tk_star, tk_slash, tk_percent,
    tk_less, tk_lessOrEqual, tk_greater, tk_greaterOrEqual,
    tk_equal, tk_notEqual,
    tk_if, tk_then, tk_elif, tk_else, tk_fi, tk_while, tk_do, tk_od,
    tk_nil
} TokenType_t;

typedef struct {
    char *rw_text;
    TokenType_t rw_code;
} ReservedWord_t;

static char *TokenNames[] = {
    "<ERROR>", "<EOF>", "<NUMBER>", "<STRING>", "<PROC>", "<ID>",
    "(", ")", "{", "}", ";", ",", ":", ".", ":=", "+", "-", "*", "/",
    "%", "<", "<=", ">", ">=", "=", "~=",
    "if", "then", "elif", "else", "fi", "while", "do", "od", "nil"
};

static ReservedWord_t ReservedWords[] = {
    {"if", tk_if},
    {"then", tk_then},
    {"elif", tk_elif},
    {"else", tk_else},
    {"fi", tk_fi},
    {"while", tk_while},
    {"do", tk_do},
    {"od", tk_od},
    {"nil", tk_nil},
    {NULL, tk_error},
};

typedef struct {
    char *ex_pointer;
    char ex_currentChar, ex_nextChar;
} ExecutionState_t;

typedef struct {
    char *bi_name;
    bool_t (*bi_func)(BUILT_IN_PROTO);
    Type_t bi_resultType;
    Type_t bi_par1Type;
    Type_t bi_par2Type;
    Type_t bi_par3Type;
} Builtin_t;

typedef struct traceBack {
    struct traceBack *tb_next;
    Thing_t *tb_thing;
    char *tb_name;
} TraceBack_t;

static void showThing(Thing_t *th);
static bool_t parseExpression(bool_t ignore, Value_t *result);
static bool_t parseSequence(bool_t ignore, Value_t *result);

static enum {Eof0, Eof1, Eof2} EofState;
static ExecutionState_t Ex;
static TokenType_t Token;
static Thing_t GlobalThing;
static Variable_t **LocalContext;
static struct Client *ActiveClient;
static Thing_t *ClientThing;
static TraceBack_t *TraceBack;
static char *LastId;
static Thing_t *LastThing;
static int IntValue;
static char StringBuffer[STRING_MAX];

static bool_t
    BI_cprint(BUILT_IN_PROTO),
    BI_aprint(BUILT_IN_PROTO),
    BI_global(BUILT_IN_PROTO),
    BI_itos(BUILT_IN_PROTO),
    BI_length(BUILT_IN_PROTO),
    BI_substr(BUILT_IN_PROTO),
    BI_strtoproc(BUILT_IN_PROTO),
    BI_random(BUILT_IN_PROTO),
    BI_word(BUILT_IN_PROTO),
    BI_gettail(BUILT_IN_PROTO),
    BI_settail(BUILT_IN_PROTO),
    BI_me(BUILT_IN_PROTO),
    BI_location(BUILT_IN_PROTO),
    BI_name(BUILT_IN_PROTO),
    BI_move(BUILT_IN_PROTO),
    BI_typeof(BUILT_IN_PROTO),
    BI_describe(BUILT_IN_PROTO),
    BI_quit(BUILT_IN_PROTO),
    BI_thing(BUILT_IN_PROTO),
    BI_destroy(BUILT_IN_PROTO),
    BI_find(BUILT_IN_PROTO),
    BI_findPlayer(BUILT_IN_PROTO),
    BI_forEachPlayer(BUILT_IN_PROTO),
    BI_createMachine(BUILT_IN_PROTO),
    BI_destroyMachine(BUILT_IN_PROTO),
    BI_after(BUILT_IN_PROTO),
    BI_force(BUILT_IN_PROTO);

static Builtin_t Builtins[] = {
    {"cprint", BI_cprint, t_void, t_thing, t_string, t_void},
    {"aprint", BI_aprint, t_void, t_thing, t_string, t_void},
    {"global", BI_global, t_thing, t_void, t_void, t_void},
    {"itos", BI_itos, t_string, t_int, t_void, t_void},
    {"length", BI_length, t_int, t_string, t_void, t_void},
    {"substr", BI_substr, t_string, t_string, t_int, t_int},
    {"strtoproc", BI_strtoproc, t_proc, t_string, t_void, t_void},
    {"random", BI_random, t_int, t_void, t_void, t_void},
    {"word", BI_word, t_string, t_void, t_void, t_void},
    {"gettail", BI_gettail, t_string, t_void, t_void, t_void},
    {"settail", BI_settail, t_void, t_string, t_void, t_void},
    {"me", BI_me, t_thing, t_void, t_void, t_void},
    {"location", BI_location, t_thing, t_thing, t_void, t_void},
    {"name", BI_name, t_string, t_thing, t_void, t_void},
    {"move", BI_move, t_void, t_thing, t_thing, t_void},
    {"typeof", BI_typeof, t_int, t_thing, t_string, t_void},
    {"describe", BI_describe, t_void, t_thing, t_void, t_void},
    {"quit", BI_quit, t_void, t_void, t_void, t_void},
    {"thing", BI_thing, t_thing, t_string, t_void, t_void},
    {"destroy", BI_destroy, t_void, t_thing, t_void, t_void},
    {"find", BI_find, t_thing, t_string, t_void, t_void},
    {"findPlayer", BI_findPlayer, t_thing, t_string, t_thing, t_void},
    {"forEachPlayer", BI_forEachPlayer, t_void, t_proc, t_void, t_void},
    {"createMachine", BI_createMachine, t_thing, t_string, t_void, t_void},
    {"destroyMachine", BI_destroyMachine, t_void, t_thing, t_void, t_void},
    {"after", BI_after, t_void, t_int, t_proc, t_void},
    {"force", BI_force, t_void, t_thing, t_proc, t_void},
    {NULL, NULL, t_void, t_void, t_void, t_void}
};

static char *TypeName[] = {
    "void", "int", "string", "proc", "thing", "<VAR>"
};

static void
printValue(Value_t *v)
{

    switch (v->v_type) {
    case t_int:
	cprintf(ActiveClient, "%d", v->v_.v_int);
	break;
    case t_string:
	cprintf(ActiveClient, "\"%s\"", v->v_.v_string);
	break;
    case t_proc:
	cprintf(ActiveClient, "`%s`", v->v_.v_string);
	break;
    case t_void:
	break;
    case t_thing:
	showThing(v->v_.v_thing);
	break;
    default:
	cprintf(ActiveClient, "????? (type %x)", v->v_type);
	break;
    }
}

static void
showThing(Thing_t *th)
{
    Variable_t *v;

    if (th == NULL) {
	cprintf(ActiveClient, "nil\n");
    } else {
	cprintf(ActiveClient, "thing %s:\n", th->th_name);
	v = th->th_context;
	if (v != NULL) {
	    while (v != NULL) {
		cprintf(ActiveClient, "%s ==> ", v->v_name);
		if (v->v_value.v_type == t_thing) {
		    if (v->v_value.v_.v_thing == NULL) {
			cprintf(ActiveClient, "nil");
		    } else {
			cprintf(ActiveClient, "<CONTEXT>");
		    }
		} else {
		    printValue(&v->v_value);
		}
		cprintf(ActiveClient, "\n");
		v = v->v_next;
	    }
	} else {
	    cprintf(ActiveClient, "Context is empty.\n");
	}
    }
}

static Variable_t *
newVariable(Variable_t **ct, char *name)
{
    Variable_t *v;

    v = (Variable_t *) mudAlloc(sizeof(Variable_t));
    v->v_next = *ct;
    *ct = v;
    v->v_name = name;
    v->v_value.v_type = t_void;
    return(v);
}

static void
eprintf(char *fmt, ...)
{
    va_list ap;
    TraceBack_t *tb;

    va_start(ap, fmt);
    va_cprintf(ActiveClient, fmt, ap);
    va_end(ap);
    tb = TraceBack;
    if (tb == NULL) {
	cprintf(ActiveClient, "error occurred at top level\n");
    } else {
	cprintf(ActiveClient, "Traceback:");
	while (tb != NULL) {
	    cprintf(ActiveClient, " ");
	    if (tb->tb_thing != NULL) {
		cprintf(ActiveClient, "%s.", tb->tb_thing->th_name);
	    }
	    if (tb->tb_name != NULL) {
		cprintf(ActiveClient, "%s", tb->tb_name);
	    } else {
		cprintf(ActiveClient, "???");
	    }
	    tb = tb->tb_next;
	}
	cprintf(ActiveClient, "\n");
    }
}

static char *
tShow(void)
{
    /* static, since we return a pointer to it */
    static char buffer[50];

    switch (Token) {
    case tk_number:
	sprintf(&buffer[0], "%d", IntValue);
	return(&buffer[0]);
    case tk_stringConstant:
	sprintf(&buffer[0], "\"%14s...\"", &StringBuffer[0]);
	return(&buffer[0]);
    case tk_procConstant:
	sprintf(&buffer[0], "`%14s...`", &StringBuffer[0]);
	return(&buffer[0]);
    case tk_id:
	return(&StringBuffer[0]);
    default:
	return(TokenNames[Token]);
    }
}

static bool_t
BI_cprint(BUILT_IN_PROTO)
{
    struct Client *cl;

    cl = findClient(par1->v_.v_thing);
    if (cl != NULL) {
	cprintf(cl, "%s", par2->v_.v_string);
    }
    mudFree(par2->v_.v_string);
    return(TRUE);
}

static bool_t
BI_aprint(BUILT_IN_PROTO)
{
    struct Client *cl;
    ExecutionState_t mainState;
    Variable_t **saveLocalContext;
    struct Client *saveActiveClient;
    Thing_t *saveClientThing;

    cl = findClient(par1->v_.v_thing);
    if (cl != NULL) {
	mainState = Ex;
	saveLocalContext = LocalContext;
	saveActiveClient = ActiveClient;
	saveClientThing = ClientThing;
	aprintf(cl, "%s", par2->v_.v_string);
	ClientThing = saveClientThing;
	ActiveClient = saveActiveClient;
	LocalContext = saveLocalContext;
	Ex = mainState;
	EofState = Eof0;
    }
    mudFree(par2->v_.v_string);
    return(TRUE);
}

static bool_t
BI_global(BUILT_IN_PROTO)
{
    result->v_.v_thing = &GlobalThing;
    return(TRUE);
}

static bool_t
BI_itos(BUILT_IN_PROTO)
{
    char buf[50];

    sprintf(&buf[0], "%d", par1->v_.v_int);
    result->v_.v_string = mudStrDup(&buf[0]);
    return(TRUE);
}

static bool_t
BI_length(BUILT_IN_PROTO)
{
    result->v_.v_int = strlen(par1->v_.v_string);
    mudFree(par1->v_.v_string);
    return(TRUE);
}

static bool_t
BI_substr(BUILT_IN_PROTO)
{
    char *p, *q;
    int i;

    p = par1->v_.v_string;
    i = par2->v_.v_int;
    while (i > 0 && *p != '\0') {
	++p;
	i -= 1;
    }
    q = p;
    i = par3->v_.v_int;
    while (i > 0 && *p != '\0') {
	++p;
	i -= 1;
    }
    *p = '\0';
    result->v_.v_string = mudStrDup(q);
    mudFree(par1->v_.v_string);
    return(TRUE);
}

/* NOTE: each call of this builtin is likely to lose storage, unless the
   result is only used to define a new value. */

static bool_t
BI_strtoproc(BUILT_IN_PROTO)
{
    *result = *par1;
    return(TRUE);
}

static bool_t
BI_random(BUILT_IN_PROTO)
{
    extern int rand(void);

    result->v_.v_int = rand();
    return(TRUE);
}

static bool_t
BI_word(BUILT_IN_PROTO)
{
    result->v_.v_string = getWord();
    return(TRUE);
}

static bool_t
BI_gettail(BUILT_IN_PROTO)
{
    result->v_.v_string = getTail();
    return(TRUE);
}

static bool_t
BI_settail(BUILT_IN_PROTO)
{
    setTail(par1->v_.v_string);
    mudFree(par1->v_.v_string);
    return(TRUE);
}

static bool_t
BI_me(BUILT_IN_PROTO)
{
    result->v_.v_thing = ClientThing;
    return(TRUE);
}

static bool_t
BI_location(BUILT_IN_PROTO)
{
    Thing_t *location;

    if (par1->v_.v_thing == NULL) {
	eprintf("*** nil thing passed to 'location'\n");
	return(FALSE);
    }
    location = clientLocation(par1->v_.v_thing);
    if (location != NULL) {
	result->v_.v_thing = location;
	return(TRUE);
    } else {
	eprintf("*** 'location' called on non-player '%s'\n",
		par1->v_.v_thing->th_name);
	return(FALSE);
    }
}

static bool_t
BI_name(BUILT_IN_PROTO)
{
    if (par1->v_.v_thing == NULL) {
	eprintf("*** nil thing passed to 'name'\n");
	return(FALSE);
    }
    result->v_.v_string = mudStrDup(par1->v_.v_thing->th_name);
    return(TRUE);
}

static bool_t
BI_move(BUILT_IN_PROTO)
{
    if (par1->v_.v_thing == NULL) {
	eprintf("*** nil first thing passed to 'move'\n");
	return(FALSE);
    }
    if (par2->v_.v_thing == NULL) {
	eprintf("*** nil second thing passed to 'move'\n");
	return(FALSE);
    }
    setClientLocation(par1->v_.v_thing, par2->v_.v_thing);
    return(TRUE);
}

static bool_t
BI_typeof(BUILT_IN_PROTO)
{
    Variable_t *v;

    if (par1->v_.v_thing == NULL) {
	eprintf("*** nil thing passed to 'typeof'\n");
	return(FALSE);
    }
    v = par1->v_.v_thing->th_context;
    while (v != NULL && strcmp(v->v_name, par2->v_.v_string) != 0) {
	v = v->v_next;
    }
    mudFree(par2->v_.v_string);
    result->v_.v_int = (v == NULL) ? -1 : v->v_value.v_type;
    return(TRUE);
}

static bool_t
BI_describe(BUILT_IN_PROTO)
{
    if (par1->v_.v_thing == NULL) {
	eprintf("*** nil thing passed to 'describe'\n");
	return(FALSE);
    }
    showThing(par1->v_.v_thing);
    return(TRUE);
}

static bool_t
BI_quit(BUILT_IN_PROTO)
{
    setQuit(ActiveClient);
    return(TRUE);
}

static bool_t
BI_thing(BUILT_IN_PROTO)
{
    if (*par1->v_.v_string != '\0' && findThing(par1->v_.v_string) != NULL) {
	eprintf("*** thing '%s' already exists\n", par1->v_.v_string);
	mudFree(par1->v_.v_string);
	return(FALSE);
    }
    if (*par1->v_.v_string == '\0') {
	mudFree(par1->v_.v_string);
	result->v_.v_thing = newThing("");
    } else {
	result->v_.v_thing = newThing(par1->v_.v_string);
    }
    return(TRUE);
}

static bool_t
BI_destroy(BUILT_IN_PROTO)
{
    if (par1->v_.v_thing == NULL) {
	eprintf("*** nil thing passed to 'destroy'\n");
	return(FALSE);
    }
    if (destroyThing(par1->v_.v_thing)) {
	return(TRUE);
    } else {
	eprintf("*** thing to 'destroy' is invalid!\n");
	return(FALSE);
    }
}

static bool_t
BI_find(BUILT_IN_PROTO)
{
    result->v_.v_thing = findThing(par1->v_.v_string);
    mudFree(par1->v_.v_string);
    return(TRUE);
}

static bool_t
BI_findPlayer(BUILT_IN_PROTO)
{
    result->v_.v_thing = findActivePlayer(par1->v_.v_string, par2->v_.v_thing);
    mudFree(par1->v_.v_string);
    return(TRUE);
}

static bool_t
BI_forEachPlayer(BUILT_IN_PROTO)
{
    ExecutionState_t mainState;
    Variable_t **saveLocalContext;
    struct Client *saveActiveClient;
    Thing_t *saveClientThing;

    mainState = Ex;
    saveLocalContext = LocalContext;
    saveActiveClient = ActiveClient;
    saveClientThing = ClientThing;
    result->v_.v_int = forEachPlayer(par1->v_.v_proc);
    ClientThing = saveClientThing;
    ActiveClient = saveActiveClient;
    LocalContext = saveLocalContext;
    Ex = mainState;
    EofState = Eof0;
    return(TRUE);
}

static bool_t
BI_createMachine(BUILT_IN_PROTO)
{
    if (*par1->v_.v_string == '\0') {
	eprintf("*** empty name passed to 'createMachine'\n");
	mudFree(par1->v_.v_string);
	return(FALSE);
    }
    result->v_.v_thing = createMachine(par1->v_.v_string);
    return(TRUE);
}

static bool_t
BI_destroyMachine(BUILT_IN_PROTO)
{
    if (par1->v_.v_thing == NULL) {
	eprintf("*** nil thing passed to 'destroyMachine'\n");
	return(FALSE);
    }
    if (! destroyMachine(par1->v_.v_thing)) {
	eprintf("*** machine not found in 'destroyMachine'\n");
	return(FALSE);
    }
    return(TRUE);
}

static bool_t
BI_after(BUILT_IN_PROTO)
{
    if (par1->v_.v_int < 0) {
	eprintf("*** negative time given to 'after'\n");
	return(FALSE);
    }
    doAfter(ActiveClient, par1->v_.v_int, par2->v_.v_proc);
    return(TRUE);
}

static bool_t
BI_force(BUILT_IN_PROTO)
{
    if (par1->v_.v_thing == NULL) {
	eprintf("*** nil thing passed to 'force'\n");
	return(FALSE);
    }
    if (! doForce(par1->v_.v_thing, par2->v_.v_proc)) {
	eprintf("*** player not found in 'force'\n");
	return(FALSE);
    }
    return(TRUE);
}

static Builtin_t *
findBuiltin(char *name)
{
    Builtin_t *bi;

    bi = &Builtins[0];
    while (bi->bi_name != NULL && strcmp(&StringBuffer[0], bi->bi_name) != 0) {
	++bi;
    }
    if (bi->bi_name == NULL) {
	return(NULL);
    } else {
	return(bi);
    }
}

static void
getChar(void)
{
    Ex.ex_currentChar = Ex.ex_nextChar;
    if (*Ex.ex_pointer == '\0') {
	if (EofState == Eof0) {
	    EofState = Eof1;
	} else {
	    EofState = Eof2;
	}
	Ex.ex_nextChar = ' ';
    } else {
	Ex.ex_nextChar = *Ex.ex_pointer;
	++Ex.ex_pointer;
    }
}

static void
skipWhiteSpace(void)
{
    while (EofState != Eof2 && isspace(Ex.ex_currentChar)) {
	getChar();
    }
}

static void
getToken(void)
{
    int n;
    ReservedWord_t *rw;

    skipWhiteSpace();
    if (EofState == Eof2) {
	Token = tk_eof;
	return;
    }
    if (isalpha(Ex.ex_currentChar)) {
	n = 0;
	while (isalnum(Ex.ex_currentChar) || Ex.ex_currentChar == '_') {
	    if (n < STRING_MAX) {
		StringBuffer[n] = Ex.ex_currentChar;
		n += 1;
	    }
	    getChar();
	}
	StringBuffer[n] = '\0';
	rw = &ReservedWords[0];
	while (rw->rw_text != NULL &&
	       strcmp(rw->rw_text, &StringBuffer[0]) != 0)
	{
	    ++rw;
	}
	if (rw->rw_text != NULL) {
	    Token = rw->rw_code;
	} else {
	    Token = tk_id;
	}
    } else if (isdigit(Ex.ex_currentChar)) {
	n = 0;
	while (isdigit(Ex.ex_currentChar)) {
	    n *= 10;
	    n += (Ex.ex_currentChar - '0');
	    getChar();
	}
	Token = tk_number;
	IntValue = n;
    } else if (Ex.ex_currentChar == '"') {
	getChar();
	n = 0;
	while (EofState != Eof2 && Ex.ex_currentChar != '"') {
	    if (Ex.ex_currentChar == '\\') {
		getChar();
		if (EofState != Eof2) {
		    if (Ex.ex_currentChar == 'n') {
			Ex.ex_currentChar = '\n';
		    }
		    if (n < STRING_MAX) {
			StringBuffer[n] = Ex.ex_currentChar;
			n += 1;
		    }
		    getChar();
		}
	    } else {
		if (n < STRING_MAX) {
		    StringBuffer[n] = Ex.ex_currentChar;
		    n += 1;
		}
		getChar();
	    }
	}
	Token = tk_stringConstant;
	if (Ex.ex_currentChar == '"') {
	    getChar();
	} else {
	    eprintf("*** unterminated string constant (char %c)\n",
		    Ex.ex_currentChar);
	}
	StringBuffer[n] = '\0';
    } else if (Ex.ex_currentChar == '`') {
	getChar();
	n = 0;
	while (EofState != Eof2 && Ex.ex_currentChar != '`') {
	    if (n < STRING_MAX) {
		StringBuffer[n] = Ex.ex_currentChar;
		n += 1;
	    }
	    getChar();
	}
	Token = tk_procConstant;
	if (Ex.ex_currentChar == '`') {
	    getChar();
	} else {
	    eprintf("*** unterminated proc constant (char %c)\n",
		    Ex.ex_currentChar);
	}
	StringBuffer[n] = '\0';
    } else {
	Token = tk_error;
	switch (Ex.ex_currentChar) {
	case '+':
	    Token = tk_plus;
	    break;
	case '-':
	    Token = tk_minus;
	    break;
	case '*':
	    Token = tk_star;
	    break;
	case '/':
	    Token = tk_slash;
	    break;
	case '%':
	    Token = tk_percent;
	    break;
	case ':':
	    if (Ex.ex_nextChar == '=') {
		getChar();
		Token = tk_assign;
	    } else {
		Token = tk_colon;
	    }
	    break;
	case ',':
	    Token = tk_comma;
	    break;
	case ';':
	    Token = tk_semi;
	    break;
	case '(':
	    Token = tk_lParen;
	    break;
	case ')':
	    Token = tk_rParen;
	    break;
	case '{':
	    Token = tk_lBrace;
	    break;
	case '}':
	    Token = tk_rBrace;
	    break;
	case '<':
	    if (Ex.ex_nextChar == '=') {
		getChar();
		Token = tk_lessOrEqual;
	    } else {
		Token = tk_less;
	    }
	    break;
	case '>':
	    if (Ex.ex_nextChar == '=') {
		getChar();
		Token = tk_greaterOrEqual;
	    } else {
		Token = tk_greater;
	    }
	    break;
	case '=':
	    Token = tk_equal;
	    break;
	case '~':
	    if (Ex.ex_nextChar == '=') {
		getChar();
		Token = tk_notEqual;
	    }
	    break;
	case '.':
	    Token = tk_dot;
	    break;
	}
	if (Token == tk_error) {
	    eprintf("invalid token starting with '%c'\n", Ex.ex_currentChar);
	}
	getChar();
    }
}

static bool_t
getOnePar(bool_t ignore, Value_t *par, bool_t needComma, Builtin_t *bi,
	  Type_t parType, char *which)
{
    if (needComma) {
	if (Token != tk_comma) {
	    eprintf("*** expecting ',' between parameters to '%s' (got %s)\n",
		    bi->bi_name, tShow());
	    return(FALSE);
	}
	getToken();
    }
    if (! parseExpression(ignore, par)) {
	return(FALSE);
    }
    if (par->v_type != parType && ! ignore) {
	cprintf(ActiveClient, "*** invalid %s parameter to '%s': ",
		which, bi->bi_name);
	cprintf(ActiveClient, "got %s value instead of %s\n",
		TypeName[par->v_type], TypeName[parType]);
	if (par->v_type == t_void) {
	    cprintf(ActiveClient, "void");
	} else {
	    printValue(par);
	}
	eprintf("\n");
	return(FALSE);
    }
    return(TRUE);
}

static bool_t
callBuiltin(Builtin_t *bi, bool_t ignore, Value_t *result)
{
    Value_t par1, par2, par3;
    bool_t resultBool;
    TraceBack_t tb;
    bool_t hadError;

    getToken();
    if (Token != tk_lParen) {
	eprintf("*** expecting '(' on call to '%s' (got %s)\n", bi->bi_name,
		tShow());
	return(FALSE);
    }
    getToken();
    par1.v_type = t_void;
    par2.v_type = t_void;
    par3.v_type = t_void;
    if (bi->bi_par1Type != t_void) {
	hadError = FALSE;
	if (! getOnePar(ignore, &par1, FALSE, bi, bi->bi_par1Type, "first")) {
	    hadError = TRUE;
	} else {
	    if (bi->bi_par2Type != t_void) {
		if (! getOnePar(ignore, &par2, TRUE, bi, bi->bi_par2Type,
				"second"))
		{
		    hadError = TRUE;
		} else {
		    if (bi->bi_par3Type != t_void) {
			if (! getOnePar(ignore, &par3, TRUE, bi,
					bi->bi_par3Type, "third"))
			{
			    hadError = TRUE;
			}
		    }
		}
	    }
	}
	if (hadError) {
	    if (! ignore) {
		if (par3.v_type == t_string) {
		    mudFree(par3.v_.v_string);
		}
		if (par2.v_type == t_string) {
		    mudFree(par2.v_.v_string);
		}
		if (par1.v_type == t_string) {
		    mudFree(par1.v_.v_string);
		}
	    }
	    return(FALSE);
	}
    }
    if (Token != tk_rParen) {
	eprintf("*** expecting ')' on call to '%s' (got %s)\n", bi->bi_name,
		tShow());
	return(FALSE);
    }
    /* want to use 'getToken' AFTER call of 'forEachPlayer' */
    if (ignore) {
	getToken();
	return(TRUE);
    }
    tb.tb_next = TraceBack;
    tb.tb_thing = NULL;
    tb.tb_name = bi->bi_name;
    TraceBack = &tb;
    resultBool = (*bi->bi_func)(&par1, &par2, &par3, result);
    TraceBack = tb.tb_next;
    getToken();
    result->v_type = bi->bi_resultType;
    return(resultBool);
}

static bool_t
parseIf(bool_t ignore, Value_t *result)
{
    bool_t doneOne;

    doneOne = FALSE;
    do {
	getToken();
	if (! parseExpression(ignore, result)) {
	    return(FALSE);
	}
	if (! ignore && result->v_type != t_int) {
	    eprintf("*** condition for 'if' must be int not %s\n",
		    TypeName[result->v_type]);
	    if (result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
	if (result->v_.v_int) {
	    doneOne = TRUE;
	}
	if (Token != tk_then) {
	    eprintf("*** expecting 'then' in 'if' (got %s)\n", tShow());
	    return(FALSE);
	}
	getToken();
	if (! parseSequence(ignore || ! doneOne, result)) {
	    return(FALSE);
	}
	if (doneOne) {
	    ignore = TRUE;
	}
    } while (Token == tk_elif);
    if (Token == tk_else) {
	getToken();
	if (! parseSequence(ignore, result)) {
	    return(FALSE);
	}
    } else {
	if (! ignore) {
	    result->v_type = t_void;
	}
    }
    if (Token == tk_fi) {
	getToken();
    } else {
	eprintf("*** expecting 'fi' to end 'if' (got %s)\n", tShow());
	return(FALSE);
    }
    return(TRUE);
}

static bool_t
parseWhile(bool_t ignore, Value_t *result)
{
    ExecutionState_t saveState;
    bool_t localIgnore;

    localIgnore = ignore;
    saveState = Ex;
    do {
	Ex = saveState;
	EofState = Eof0;
	getToken();
	if (! parseSequence(localIgnore, result)) {
	    return(FALSE);
	}
	if (! localIgnore && result->v_type != t_int) {
	    eprintf("*** condition for 'while' must be int, not %s\n",
		    TypeName[result->v_type]);
	    if (result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
	if (! result->v_.v_int) {
	    localIgnore = TRUE;
	}
	if (Token == tk_do) {
	    getToken();
	} else {
	    eprintf("*** expecting 'do' in 'while' (got %s)\n", tShow());
	    return(FALSE);
	}
	if (! parseSequence(localIgnore, result)) {
	    return(FALSE);
	}
	if (! localIgnore && result->v_type != t_void) {
	    eprintf("*** body of 'while' must be void, not %s\n",
		    TypeName[result->v_type]);
	    if (result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
    } while (! localIgnore);
    if (Token == tk_od) {
	getToken();
    } else {
	eprintf("*** expecting 'od' to end 'while' (got %s)\n", tShow());
	return(FALSE);
    }
    if (! ignore) {
	result->v_type = t_void;
    }
    return(TRUE);
}

static bool_t
parseUnit(bool_t ignore, Value_t *result)
{
    Variable_t *v;
    Builtin_t *bi;

    switch (Token) {
    case tk_if:
	return(parseIf(ignore, result));
    case tk_while:
	return(parseWhile(ignore, result));
    case tk_nil:
	if (! ignore) {
	    result->v_type = t_thing;
	    result->v_.v_thing = NULL;
	}
	getToken();
	break;
    case tk_lParen:
	getToken();
	if (! parseExpression(ignore, result)) {
	    return(FALSE);
	}
	if (Token != tk_rParen) {
	    eprintf("*** missing closing ')' after subexpression (got %s)\n",
		    tShow());
	    return(FALSE);
	}
	getToken();
	break;
    case tk_lBrace:
	getToken();
	if (! parseExpression(ignore, result)) {
	    return(FALSE);
	}
	if (Token != tk_rBrace) {
	    eprintf("*** missing closing '}' after dynamic symbol (got %s)\n",
		    tShow());
	    if (result->v_type == t_string && ! ignore) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
	if (! ignore) {
	    if (result->v_type != t_string) {
		eprintf("*** result of {...} must be string not %s\n",
			TypeName[result->v_type]);
		return(FALSE);
	    }
	    if (LocalContext != NULL) {
		v = *LocalContext;
		while (v != NULL &&
		       strcmp(v->v_name, result->v_.v_string) != 0)
		{
		    v = v->v_next;
		}
	    } else {
		v = NULL;
	    }
	    if (v == NULL) {
		v = GlobalThing.th_context;
		while (v != NULL &&
		       strcmp(v->v_name, result->v_.v_string) != 0)
		{
		    v = v->v_next;
		}
	    }
	    if (v == NULL) {
		eprintf("*** '%s' is not defined (dynamic)\n",
			result->v_.v_string);
		mudFree(result->v_.v_string);
		return(FALSE);
	    } else {
		LastThing = NULL;
		LastId = v->v_name;
	    }
	    mudFree(result->v_.v_string);
	    *result = v->v_value;
	    if (result->v_type == t_string) {
		result->v_.v_string = mudStrDup(result->v_.v_string);
	    }
	}
	getToken();
	break;
    case tk_number:
	if (! ignore) {
	    result->v_type = t_int;
	    result->v_.v_int = IntValue;
	}
	getToken();
	break;
    case tk_stringConstant:
	if (! ignore) {
	    result->v_type = t_string;
	    result->v_.v_string = mudStrDup(&StringBuffer[0]);
	}
	getToken();
	break;
    case tk_procConstant:
	if (! ignore) {
	    result->v_type = t_proc;
	    result->v_.v_string = mudStrDup(&StringBuffer[0]);
	}
	getToken();
	break;
    case tk_id:
	bi = findBuiltin(&StringBuffer[0]);
	if (bi != NULL) {
	    return(callBuiltin(bi, ignore, result));
	}
	if (! ignore) {
	    v = NULL;
	    if (LocalContext != NULL) {
		v = *LocalContext;
		while (v != NULL && strcmp(v->v_name, &StringBuffer[0]) != 0) {
		    v = v->v_next;
		}
	    }
	    if (v == NULL) {
		v = GlobalThing.th_context;
		while (v != NULL && strcmp(v->v_name, &StringBuffer[0]) != 0) {
		    v = v->v_next;
		}
	    }
	    skipWhiteSpace();
	    if (Ex.ex_currentChar == ':' && Ex.ex_nextChar == '=') {
		if (v == NULL) {
		    if (LocalContext == NULL) {
			v = newVariable(&GlobalThing.th_context,
					mudStrDup(&StringBuffer[0]));
		    } else {
			v = newVariable(LocalContext,
					mudStrDup(&StringBuffer[0]));
		    }
		}
		result->v_type = t_variable;
		result->v_.v_variable = v;
	    } else {
		if (v == NULL) {
		    eprintf("*** variable '%s' is not defined\n",
			    &StringBuffer[0]);
		    return(FALSE);
		}
		LastThing = NULL;
		LastId = v->v_name;
		*result = v->v_value;
		if (result->v_type == t_string) {
		    result->v_.v_string = mudStrDup(result->v_.v_string);
		}
	    }
	}
	getToken();
	break;
    default:
	eprintf("*** unknown syntax at token %s\n", tShow());
	getToken();
	break;
    }
    return(TRUE);
}

static void
freeLocals(Variable_t *locals)
{
    Variable_t *v;

    while (locals != NULL) {
	v = locals;
	locals = locals->v_next;
	if (v->v_value.v_type == t_string) {
	    mudFree(v->v_value.v_.v_string);
	}
	mudFree(v->v_name);
	mudFree(v);
    }
}

static bool_t
doCall(bool_t ignore, Value_t *result)
{
    Variable_t **saveLocalContext;
    Variable_t *v, *locals;
    Variable_t **lastLocal;
    ExecutionState_t mainState, procState;
    Value_t val;
    TraceBack_t tb;
    bool_t bodyResult;

    if (! ignore && result->v_type != t_proc) {
	eprintf("*** can only call procs (got %s)\n",TypeName[result->v_type]);
	return(FALSE);
    }
    locals = NULL;
    if (! ignore) {
	mainState = Ex;
	Ex.ex_pointer = result->v_.v_string;
	Ex.ex_currentChar = ' ';
	Ex.ex_nextChar = ' ';
	EofState = Eof0;
	getToken();
	if (Token != tk_lParen) {
	    eprintf("*** called proc does not start with '(' (got %s)\n",
		    tShow());
	    return(FALSE);
	}
	getToken();
	lastLocal = &locals;
	while (Token != tk_rParen) {
	    if (Token != tk_id) {
		eprintf("*** expecting proc parameter name (got %s)\n",
			tShow());
		freeLocals(locals);
		return(FALSE);
	    }
	    v = (Variable_t *) mudAlloc(sizeof(Variable_t));
	    v->v_next = NULL;
	    v->v_name = mudStrDup(&StringBuffer[0]);
	    v->v_value.v_type = t_void;
	    *lastLocal = v;
	    lastLocal = &v->v_next;
	    getToken();
	    if (Token == tk_comma) {
		getToken();
	    } else if (Token != tk_rParen) {
		eprintf(
	     "*** expecting ',' or ')' after formal parameter name (got %s)\n",
			tShow());
		freeLocals(locals);
		return(FALSE);
	    }
	}
	procState = Ex;
	Ex = mainState;
	EofState = Eof0;
    }
    getToken();
    tb.tb_next = TraceBack;
    tb.tb_thing = LastThing;
    tb.tb_name = LastId;
    TraceBack = &tb;
    v = locals;
    while (Token != tk_rParen) {
	if (! ignore) {
	    if (v == NULL) {
		eprintf("*** too many parameters to proc\n");
		freeLocals(locals);
		TraceBack = tb.tb_next;
		return(FALSE);
	    }
	}
	if (! parseExpression(ignore, &val)) {
	    freeLocals(locals);
	    TraceBack = tb.tb_next;
	    return(FALSE);
	}
	if (! ignore) {
	    v->v_value = val;
	    v = v->v_next;
	}
	if (Token == tk_comma) {
	    getToken();
	} else if (Token != tk_rParen) {
	    eprintf(
		  "*** expecting ',' or ')' after actual parameter (got %s)\n",
		    tShow());
	    freeLocals(locals);
	    TraceBack = tb.tb_next;
	    return(FALSE);
	}
    }
    bodyResult = TRUE;
    if (! ignore) {
	if (v != NULL) {
	    eprintf("*** too few parameters to proc\n");
	    bodyResult = FALSE;
	} else {
	    saveLocalContext = LocalContext;
	    LocalContext = &locals;
	    mainState = Ex;
	    Ex = procState;
	    EofState = Eof0;
	    getToken();
	    bodyResult = parseSequence(FALSE, result);
	    LocalContext = saveLocalContext;
	    Ex = mainState;
	    EofState = Eof0;
	}
    }
    TraceBack = tb.tb_next;
    freeLocals(locals);
    getToken();
    return(bodyResult);
}

static bool_t
parseCallOrSelect(bool_t ignore, Value_t *result)
{
    Thing_t *th;
    Variable_t *v;
    char *name;
    bool_t nameIsExpr, wasFancy;

    if (! parseUnit(ignore, result)) {
	return(FALSE);
    }
    while (Token == tk_dot || Token == tk_lParen) {
	if (Token == tk_lParen) {
	    if (! doCall(ignore, result)) {
		return(FALSE);
	    }
	} else {
	    if (result->v_type != t_thing && ! ignore) {
		eprintf("*** can only select from thing, not %s\n",
			TypeName[result->v_type]);
		return(FALSE);
	    }
	    th = result->v_.v_thing;
	    if (th == NULL && ! ignore) {
		eprintf("*** cannot select from nil\n");
		return(FALSE);
	    }
	    LastThing = th;
	    getToken();
	    if (Token == tk_lBrace) {
		wasFancy = TRUE;
		getToken();
		if (! parseExpression(ignore, result)) {
		    return(FALSE);
		}
		if (Token != tk_rBrace) {
		    eprintf(
		    "*** expecting closing '}' after property expr (got %s)\n",
			    tShow());
		    if (! ignore && result->v_type == t_string) {
			mudFree(result->v_.v_string);
		    }
		    return(FALSE);
		}
		if (result->v_type != t_string && ! ignore) {
		    eprintf("*** property expression must be string, not %s\n",
			    TypeName[result->v_type]);
		    return(FALSE);
		}
		getToken();
		nameIsExpr = TRUE;
		name = result->v_.v_string;
	    } else {
		wasFancy = FALSE;
		if (Token != tk_id) {
		    eprintf("*** expecting id after '.' (got %s)\n", tShow());
		    return(FALSE);
		}
		nameIsExpr = FALSE;
		name = &StringBuffer[0];
	    }
	    if (! ignore) {
		v = th->th_context;
		while (v != NULL) {
		    if (strcmp(v->v_name, name) == 0) {
			break;
		    }
		    v = v->v_next;
		}
		skipWhiteSpace();
		if (Ex.ex_currentChar == ':' && Ex.ex_nextChar == '=') {
		    if (v == NULL) {
			if (! nameIsExpr) {
			    name = mudStrDup(name);
			}
			v = newVariable(&th->th_context, name);
			nameIsExpr = FALSE;
		    }
		    result->v_type = t_variable;
		    result->v_.v_variable = v;
		} else {
		    if (v == NULL) {
			eprintf(
			       "*** property '%s' not defined in thing '%s'\n",
				name, th->th_name);
			return(FALSE);
		    }
		    LastId = v->v_name;
		    *result = v->v_value;
		    if (result->v_type == t_string) {
			result->v_.v_string = mudStrDup(result->v_.v_string);
		    }
		}
	    }
	    if (nameIsExpr && ! ignore) {
		mudFree(name);
	    }
	    if (! wasFancy) {
		getToken();
	    }
	}
    }
    return(TRUE);
}

static bool_t
parseMulDiv(bool_t ignore, Value_t *result)
{
    Value_t right;
    TokenType_t saveToken;

    if (! parseCallOrSelect(ignore, result)) {
	return(FALSE);
    }
    while (Token == tk_star || Token == tk_slash || Token == tk_percent) {
	saveToken = Token;
	if (! ignore && result->v_type != t_int) {
	    eprintf("*** operand to '*', '/' or '%'must be int, not %s\n",
		    TypeName[result->v_type]);
	    if (result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
	getToken();
	if (! parseCallOrSelect(ignore, &right)) {
	    return(FALSE);
	}
	if (! ignore && right.v_type != t_int) {
	    eprintf("*** operand to '*', '/' or '%' must be int, not %s\n",
		    TypeName[right.v_type]);
	    if (right.v_type == t_string) {
		mudFree(right.v_.v_string);
	    }
	    return(FALSE);
	}
	if (! ignore) {
	    switch (saveToken) {
	    case tk_star:
		result->v_.v_int = result->v_.v_int * right.v_.v_int;
		break;
	    case tk_slash:
		if (right.v_.v_int == 0) {
		    eprintf("*** division by zero\n");
		    return(FALSE);
		}
		result->v_.v_int = result->v_.v_int / right.v_.v_int;
		break;
	    case tk_percent:
		if (right.v_.v_int == 0) {
		    eprintf("*** modulo by zero\n");
		    return(FALSE);
		}
		result->v_.v_int = result->v_.v_int % right.v_.v_int;
		break;
	    default:
		mudAbort("impossible token in parseMulDivMod");
	    }
	}
    }
    return(TRUE);
}

static bool_t
parsePlusMinus(bool_t ignore, Value_t *result)
{
    Value_t right;
    int len;
    char *newString;
    bool_t isPlus;

    if (! parseMulDiv(ignore, result)) {
	return(FALSE);
    }
    while (Token == tk_plus || Token == tk_minus) {
	isPlus = (Token == tk_plus);
	if (! ignore &&
	    ((result->v_type != t_int && result->v_type != t_string) ||
	     (result->v_type == t_string && ! isPlus)))
	{
	    eprintf("*** invalid left operand to '+' or '-'\n");
	    if (result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
	getToken();
	if (! parseMulDiv(ignore, &right)) {
	    if (! ignore && result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
	if (! ignore && right.v_type != result->v_type) {
	    eprintf("*** invalid right operand to '+' or '-'\n");
	    if (right.v_type == t_string) {
		mudFree(right.v_.v_string);
	    }
	    if (result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
	if (! ignore) {
	    if (isPlus) {
		if (result->v_type == t_int) {
		    result->v_.v_int = result->v_.v_int + right.v_.v_int;
		} else {
		    len = strlen(result->v_.v_string);
		    newString =
			(char *) mudAlloc(len + strlen(right.v_.v_string) +
					  sizeof(char));
		    strcpy(newString, result->v_.v_string);
		    mudFree(result->v_.v_string);
		    strcpy(newString + len, right.v_.v_string);
		    mudFree(right.v_.v_string);
		    result->v_.v_string = newString;
		}
	    } else {
		result->v_.v_int = result->v_.v_int - right.v_.v_int;
	    }
	}
    }
    return(TRUE);
}

static bool_t
parseComparison(bool_t ignore, Value_t *result)
{
    Value_t right;
    int value;
    TokenType_t tk;
					
    if (! parsePlusMinus(ignore, result)) {
	return(FALSE);
    }
    if (Token == tk_equal || Token == tk_notEqual ||
	Token == tk_less || Token == tk_lessOrEqual ||
	Token == tk_greater || Token == tk_greaterOrEqual)
    {
	if (! ignore && result->v_type == t_void) {
	    eprintf("*** can't compare void\n");
	    return(FALSE);
	}
	tk = Token;
	if (! ignore &&
	    (result->v_type == t_proc || result->v_type == t_thing) &&
	    tk != tk_equal && tk != tk_notEqual)
	{
	    eprintf("*** can only compare procs and things for equality\n");
	    return(FALSE);
	}
	getToken();
	if (! parsePlusMinus(ignore, &right)) {
	    if (! ignore && result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
	if (! ignore && right.v_type != result->v_type) {
	    cprintf(ActiveClient,"*** incompatible operands for comparison\n");
	    eprintf("*** left is %s, but right is %s\n",
		    TypeName[result->v_type], TypeName[right.v_type]);
	    if (result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    if (right.v_type == t_string) {
		mudFree(right.v_.v_string);
	    }
	    return(FALSE);
	}
	if (! ignore) {
	    value = 0;	/* hush up gcc */
	    switch (result->v_type) {
	    case t_int:
	    case t_proc:
	    case t_thing:
		/* Lie, cheat and steal - should have separate code for
		   proc and thing, and let compiler merge them. */
		switch (tk) {
		case tk_equal:
		    value = (result->v_.v_int == right.v_.v_int);
		    break;
		case tk_notEqual:
		    value = (result->v_.v_int != right.v_.v_int);
		    break;
		case tk_less:
		    value = (result->v_.v_int < right.v_.v_int);
		    break;
		case tk_lessOrEqual:
		    value = (result->v_.v_int <= right.v_.v_int);
		    break;
		case tk_greater:
		    value = (result->v_.v_int > right.v_.v_int);
		    break;
		case tk_greaterOrEqual:
		    value = (result->v_.v_int >= right.v_.v_int);
		    break;
		default:
		    mudAbort("impossible token in parseComparison (1)");
		}
		break;
	    case t_string:
		switch (tk) {
		case tk_equal:
		    value =
			(strcmp(result->v_.v_string, right.v_.v_string) == 0);
		    break;
		case tk_notEqual:
		    value =
			(strcmp(result->v_.v_string, right.v_.v_string) != 0);
		    break;
		case tk_less:
		    value =
			(strcmp(result->v_.v_string, right.v_.v_string) == -1);
		    break;
		case tk_lessOrEqual:
		    value =
			(strcmp(result->v_.v_string, right.v_.v_string) != 1);
		    break;
		case tk_greater:
		    value =
			(strcmp(result->v_.v_string, right.v_.v_string) == 1);
		    break;
		case tk_greaterOrEqual:
		    value =
			(strcmp(result->v_.v_string, right.v_.v_string) != -1);
		    break;
		default:
		    mudAbort("impossible token in parseComparison (2)");
		}
		mudFree(result->v_.v_string);
		mudFree(right.v_.v_string);
		break;
	    default:
		mudAbort("impossible type in parseComparison");
	    }
	    result->v_type = t_int;
	    result->v_.v_int = value;
	}
    }
    return(TRUE);
}

static bool_t
parseExpression(bool_t ignore, Value_t *result)
{
    return(parseComparison(ignore, result));
}

static bool_t
parseAssignment(bool_t ignore, Value_t *result)
{
    Variable_t *v;

    if (! parseExpression(ignore, result)) {
	return(FALSE);
    }
    if (Token == tk_assign) {
	if (! ignore && result->v_type != t_variable) {
	    eprintf("*** cannot assign to this (type %s)\n",
		    TypeName[result->v_type]);
	    return(FALSE);
	}
	v = result->v_.v_variable;
	getToken();
	if (! parseExpression(ignore, result)) {
	    return(FALSE);
	}
	if (! ignore) {
	    if (result->v_type != v->v_value.v_type &&
		v->v_value.v_type != t_void)
	    {
		cprintf(ActiveClient,
			"*** assignment incompatibility for '%s'\n",
			v->v_name);
		eprintf("*** variable is %s, value is %s\n",
			TypeName[v->v_value.v_type], TypeName[result->v_type]);
		return(FALSE);
	    }
	    if (v->v_value.v_type == t_string) {
		mudFree(v->v_value.v_.v_string);
	    }
	    v->v_value = *result;
	    result->v_type = t_void;
	}
    }
    return(TRUE);
}

static bool_t
isStatementEnd(void)
{
    return(Token == tk_elif || Token == tk_else || Token == tk_fi ||
	   Token == tk_do || Token == tk_od || Token == tk_eof);
}

static bool_t
parseSequence(bool_t ignore, Value_t *result)
{

    while (1) {
	if (isStatementEnd()) {
	    if (! ignore) {
		result->v_type = t_void;
	    }
	    return(TRUE);
	}
	if (! parseAssignment(ignore, result)) {
	    return(FALSE);
	}
	if (isStatementEnd()) {
	    break;
	}
	if (! ignore && result->v_type != t_void) {
	    eprintf("*** %s value being discarded\n",TypeName[result->v_type]);
	    if (result->v_type == t_string) {
		mudFree(result->v_.v_string);
	    }
	    return(FALSE);
	}
	if (Token == tk_semi) {
	    getToken();
	} else {
	    eprintf("*** expecting ';' in sequence (got %s)\n", tShow());
	    return(FALSE);
	}
    }
    return(TRUE);
}

void
initInterpreter(Variable_t *globals)
{
    GlobalThing.th_name = mudStrDup("<GLOBALS>");
    GlobalThing.th_context = globals;
    ActiveClient = NULL;
    TraceBack = NULL;
}

Variable_t *
getGlobals(void)
{
    return(GlobalThing.th_context);
}

void
runProc(struct Client *who, Player_t *pl, char *proc)
{
    Value_t result;
    struct Client *saveActive;

    saveActive = ActiveClient;
    Ex.ex_pointer = proc;
    Ex.ex_currentChar = ' ';
    Ex.ex_nextChar = ' ';
    EofState = Eof0;
    getToken();
    ActiveClient = who;
    LocalContext = NULL;
    ClientThing = &pl->pl_thing;
    TraceBack = NULL;
    if (parseSequence(FALSE, &result)) {
	if (result.v_type != t_void) {
	    cprintf(who, "==> ");
	    printValue(&result);
	    cprintf(who, "\n");
	}
	if (result.v_type == t_string) {
	    mudFree(result.v_.v_string);
	}
    }
    ActiveClient = saveActive;
}

static void
runWithPar(struct Client *who, Player_t *pl, Value_t *parm, char *name,
	   char *proc)
{
    Value_t result;
    Variable_t *parameter, *parameters;

    ActiveClient = who;
    TraceBack = NULL;
    Ex.ex_pointer = proc;
    Ex.ex_currentChar = ' ';
    Ex.ex_nextChar = ' ';
    EofState = Eof0;
    getToken();
    if (Token != tk_lParen) {
	eprintf("*** global '%s' does not start with '(' (got %s)\n", name,
		tShow());
	return;
    }
    getToken();
    if (Token != tk_id) {
	eprintf("*** global '%s' has no parameter\n", name);
	return;
    }
    parameters = NULL;
    parameter = newVariable(&parameters, mudStrDup(&StringBuffer[0]));
    parameter->v_value = *parm;
    getToken();
    if (Token != tk_rParen) {
	eprintf("*** global '%s' does not have closing ')' (got %s)\n", name,
		tShow());
	freeLocals(parameters);
	return;
    }
    getToken();
    LocalContext = &parameters;
    ClientThing = &pl->pl_thing;
    if (parseSequence(FALSE, &result)) {
	if (result.v_type != t_void) {
	    eprintf("*** global '%s' returned %s, not void\n", name,
		    TypeName[result.v_type]);
	}
	if (result.v_type == t_string) {
	    mudFree(result.v_.v_string);
	}
    }
    freeLocals(parameters);
}

static void
runNamedWithPar(struct Client *who, Player_t *pl, Value_t *parm, char *name)
{
    Variable_t *v;

    v = pl->pl_thing.th_context;
    while (v != NULL && strcmp(v->v_name, name) != 0) {
	v = v->v_next;
    }
    if (v == NULL) {
	v = GlobalThing.th_context;
	while (v != NULL && strcmp(v->v_name, name) != 0) {
	    v = v->v_next;
	}
    }
    if (v != NULL) {
	if (v->v_value.v_type != t_proc) {
	    cprintf(who, "*** global '%s' must be a proc, not %s\n", name,
		    TypeName[v->v_value.v_type]);
	    return;
	}
	runWithPar(who, pl, parm, name, v->v_value.v_.v_proc);
    }
}

void
runWithPlayer(struct Client *who, Player_t *pl, char *name)
{
    Value_t v;

    v.v_type = t_thing;
    v.v_.v_thing = &pl->pl_thing;
    runNamedWithPar(who, pl, &v, name);
}

void
runWithString(struct Client *who, Player_t *pl, char *string, char *name)
{
    Value_t v;

    v.v_type = t_string;
    v.v_.v_string = mudStrDup(string);
    runNamedWithPar(who, pl, &v, name);
}

void
runExplicitWithPlayer(struct Client *who, Player_t *pl, char *proc)
{
    Value_t v;

    v.v_type = t_thing;
    v.v_.v_thing = &pl->pl_thing;
    runWithPar(who, pl, &v, "<dynamic>", proc);
}
