/*
 * net.c - code for dealing with sockets. Uses 'select'. Does non-blocking
 *	sockets for both reading and writing. Registers multiple possible
 *	client-types that do further processing (e.g. telnet).
 *	- Chris Gray, cg@ami-cg.GraySage.Edmonton.AB.CA, Nov/Dec 1998.
 */

/*
 * This code is provided as-is. No warranty of any kind is given.
 */

/*
 * As of this writing, only compilation under Linux has been tested.
 * More will come.
 */

#if defined(__unix) || defined(AMIGA)

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#if defined(AIX)
#include <sys/select.h>
#endif
#if defined(SOLARIS)
#include <sys/filio.h>
#endif
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>

typedef int Socket_t;
#define INVALID_SOCKET	-1

#elif defined(_WIN32)

/* Can redefine this before including the header file, to change the
   number of sockets we can have at once in a select call. 64 is
   the default. */

#define FD_SETSIZE	64

#include <winsock2.h>
#include <wininet.h>

typedef SOCKET Socket_t;

#else
%%%% No environment specified

#endif

#if defined(AMIGA)
#include <proto/dos.h>
extern int fcntl(int fd, int cmd, int arg);
#endif

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>

#include "types.h"
#include "telnetFlags.h"
#include "netClient.h"

#define ERRORS_TO_STDERR	1
#define DEBUG_ENABLED		1
#define DO_ABORTS		0

/*
 * This type is used to keep track of the kinds of custom client types
 * that have been registered with us. Until we get at least one, we
 * won't even have an sockets to work with.
 */

typedef struct ClientType {
    struct ClientType *clt_next;
    struct ClientType *clt_prev;
    NetClient_t *clt_netClient;
    unsigned short int clt_port;
    int clt_auxData;
    int clt_socket;
} ClientType_t;

/*
 * The actual definition of the GenericClient structure.
 */

typedef struct GenericClient {
    struct GenericClient *gcl_next;
    struct GenericClient *gcl_prev;
    const ClientType_t *gcl_type;
    Socket_t gcl_socket;
    const void *gcl_clientData;
    bool_t gcl_isShutDown;
} GenericClient_t;

static ClientType_t *ClientTypes;	/* list of custom client types */
static fd_set AllSocketBits;		/* all current sockets */
static fd_set ClientWriteBits;		/* all write-blocked sockets */
static Socket_t MainSocketMax;		/* max of just accept sockets */
static Socket_t MaxFd;			/* largest, used to bound select */
static GenericClient_t *Clients;	/* list of all clients */
static unsigned long int NextId;	/* next unique client ID */
static bool_t QuitRequested;
static NetGeneric_t TheNetGeneric;	/* the only one of these. */

#if defined(AMIGA)
static long int MyAddr;
#elif defined(_WIN32)
static unsigned int MyAddr;
#else
static int MyAddr;
#endif

#if defined(_WIN32)
static WSADATA WSAData;
#endif

#if DEBUG_ENABLED
#define DEBUG(x)    printf x
#else
#define DEBUG(x)
#endif

/*
 * myPerror - print to somewhere, an error regarding some operation.
 *	Note that all of our calls are from socket call errors.
 */

#define ERROR_BUFF_LEN	1000

#if defined(__unix) || defined(AMIGA)

static void
myPerror(const char *message)
{
    char buffer[ERROR_BUFF_LEN];
    char *reason;

#if defined(SUNOS4)
    extern int sys_nerr;
    extern char *sys_errlist[];

    if (errno >= sys_nerr) {
	reason = NULL;
    } else {
	reason = sys_errlist[errno];
    }
#else
    reason = strerror(errno);
#endif
    if (reason == NULL) {
	sprintf(&buffer[0], "%s: unknown error %d", message, errno);
    } else {
	sprintf(&buffer[0], "%s: %s", message, reason);
    }
#if ERRORS_TO_STDERR
    fprintf(stderr, "%s\n", &buffer[0]);
#else
    %%%%
#endif
}

#elif defined(_WIN32)

static void
myPerror(const char *message)
{
    HMODULE wsMod;
    DWORD mLen, len, errorCode;
    char buffer[ERROR_BUFF_LEN];

    errorCode = WSAGetLastError();
    sprintf(&buffer[0], "%s: ", message);
    mLen = strlen(&buffer[0]);
    wsMod = GetModuleHandle("wsock32");
    len =
	FormatMessage(FORMAT_MESSAGE_FROM_HMODULE | FORMAT_MESSAGE_FROM_SYSTEM,
		      (LPCVOID) wsMod, errorCode, 0, &buffer[mLen],
		      ERROR_BUFF_LEN - mLen - 1, NULL);
    if (len == 0) {
	sprintf(&buffer[mLen], "FormatMessage failed reason %d, orig error %d",
		GetLastError(), errorCode);
    } else {
	/* trim any trailing CR/LF */
	mLen = strlen(&buffer[0]);
	if (buffer[mLen - 1] == '\n') {
	    buffer[mLen - 1] = '\0';
	    mLen -= 1;
	}
	if (buffer[mLen - 1] == '\r') {
	    buffer[mLen - 1] = '\0';
	}
    }
#if ERRORS_TO_STDERR
    fprintf(stderr, "%s\n", &buffer[0]);
#else
    %%%%
#endif
}

#undef ERROR_BUFF_LEN

#else
%%%%
#endif

/*
 * killSocket - get rid of a socket.
 */

static void
killSocket(Socket_t s)
{

    shutdown(s, 2);
#if defined(_WIN32)
    closesocket(s);
#elif defined(AMIGA)
    CloseSocket(s);
#else
    close(s);
#endif
}

/*
 * deleteClient - delete this client. We are done with it.
 */

static void
deleteClient(GenericClient_t *gcl)
{

    (*gcl->gcl_type->clt_netClient->nc_free)((void *) gcl->gcl_clientData);
    if (gcl->gcl_prev != NULL) {
	gcl->gcl_prev->gcl_next = gcl->gcl_next;
    } else {
	Clients = gcl->gcl_next;
    }
    if (gcl->gcl_next != NULL) {
	gcl->gcl_next->gcl_prev = gcl->gcl_prev;
    }
    free(gcl);
}

/*
 * detachSocket - detach the client from the socket.
 */

static void
detachSocket(GenericClient_t *gcl)
{
    Socket_t s, max;

    s = gcl->gcl_socket;
    if (s != INVALID_SOCKET) {
	gcl->gcl_socket = INVALID_SOCKET;
	FD_CLR(s, &AllSocketBits);
	FD_CLR(s, &ClientWriteBits);
	killSocket(s);
	if (s == MaxFd) {
	    /* This was the biggest, and is now gone. Find new biggest. */
	    max = MainSocketMax;
	    gcl = Clients;
	    while (gcl != NULL) {
		if (gcl->gcl_socket > max) {
		    max = gcl->gcl_socket;
		}
		gcl = gcl->gcl_next;
	    }
	    MaxFd = max;
	}
    }
}

/*
 * abortClient - do everything needed to make a client go away.
 */

static void
abortClient(GenericClient_t *gcl)
{

    DEBUG(("abortClient\n"));
    if (! gcl->gcl_isShutDown) {
	/* Avoid recursion here, just in case! */
	gcl->gcl_isShutDown = TRUE;
	(*gcl->gcl_type->clt_netClient->nc_stop)
	    ((void *) gcl->gcl_clientData, FALSE);
    }
    detachSocket(gcl);
    deleteClient(gcl);
}

/*
 * setupSocket - create a socket, and bind it to the given port.
 */

static bool_t
setupSocket(int *pSocket, unsigned short int port)
{
    struct sockaddr_in myAddr;
    Socket_t s;
    long int flag;

    /* Create a non-blocking socket */
    flag = 1;
#if defined(_WIN32)
    s = WSASocket(AF_INET, SOCK_STREAM, 0, NULL, 0, 0);
    if (s == INVALID_SOCKET) {
	myPerror("can't create server socket");
    }
    if (ioctlsocket(s, FIONBIO, (unsigned long *) &flag) == SOCKET_ERROR) {
	myPerror("ioctlsocket on server socket");
	killSocket(s);
	return FALSE;
    }
#else
    s = socket(AF_INET, SOCK_STREAM, 0);
    if (s < 0) {
	myPerror("can't create server socket");
	return FALSE;
    }
    if (ioctl(s, FIONBIO, (char *) &flag) == -1) {
	myPerror("ioctl on server socket");
	killSocket(s);
	return FALSE;
    }
#endif

    /* Do this *before* the bind/listen */
#if defined(_WIN32)
    if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *) &flag, sizeof(flag))
	== SOCKET_ERROR)
#else
    if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *) &flag, sizeof(flag))
	!= 0)
#endif
    {
	myPerror("setsockopt(SO_REUSEADDR)");
	killSocket(s);
	return FALSE;
    }

    /* Bind it to the required port. */
    memset(&myAddr, 0, sizeof(struct sockaddr_in));
    myAddr.sin_port = htons(port);
    myAddr.sin_family = AF_INET;
#if defined(_WIN32)
    if (bind(s, (struct sockaddr *) &myAddr, sizeof(struct sockaddr_in)) ==
	SOCKET_ERROR)
#else
    if (bind(s, (struct sockaddr *) &myAddr, sizeof(struct sockaddr_in)) < 0)
#endif
    {
	myPerror("bind");
	killSocket(s);
	return FALSE;
    }

    /* Accept connections on it. */
#if defined(_WIN32)
    if (listen(s, 5) == SOCKET_ERROR)
#else
    if (listen(s, 5) < 0)
#endif
    {
	myPerror("listen");
	killSocket(s);
	return FALSE;
    }

    /* Add it to the set of sockets our 'select' pays attention to. */
    FD_SET(s, &AllSocketBits);
    if (s > MainSocketMax) {
	MainSocketMax = s;
    }
    if (s > MaxFd) {
	MaxFd = s;
    }
    *pSocket = s;
    return TRUE;
}

/*
 * netAddClientType - this is the externally available interface by which
 *	other code can ask us to support a new type of client. They give
 *	us a vector of functions to call for various things, the port to
 *	use, and an int worth of other data. The telnet client type uses
 *	that int as initial flag settings and negotiation options.
 */

bool_t
netAddClientType(NetClient_t *netClient, unsigned short int port, int auxData)
{
    ClientType_t *clt;

    clt = malloc(sizeof(ClientType_t));
    if (clt == NULL) {
	return FALSE;
    }
    clt->clt_next = ClientTypes;
    clt->clt_netClient = netClient;
    clt->clt_port = port;
    clt->clt_auxData = auxData;
    if (! setupSocket(&clt->clt_socket, port)) {
	free(clt);
	return FALSE;
    }
    ClientTypes = clt;
    DEBUG(("added client type on port %d auxData 0x%x\n", port, auxData));
    return TRUE;
}

/*
 * netWrite - write some data directly to a client socket.
 */

int
netWrite(GenericClient_t *gcl, const void *buf, int len)
{
    int res;

    res = send(gcl->gcl_socket, buf, len, 0);
    if (res < 0) {
#if defined(_WIN32)
	if (WSAGetLastError() != WSAEWOULDBLOCK)
#else
	if (errno != EWOULDBLOCK)
#endif
	{
	    /* Something wrong - shutdown the client. */
	    abortClient(gcl);
	} else {
	    /* Fake this, so caller will not think we have died. */
	    res = 0;
	}
    }
    return res;
}

/*
 * netRead - read some data directly from a client socket.
 */

int
netRead(GenericClient_t *gcl, void *buf, int len)
{
    int res;

    res = recv(gcl->gcl_socket, buf, len, 0);
    if (res <= 0) {
#if defined(_WIN32)
	if (WSAGetLastError() != WSAEWOULDBLOCK)
#else
	if (errno != EWOULDBLOCK)
#endif
	{
	    /* Something wrong with the client. */
	    abortClient(gcl);
	}
    }
    return res;
}

/*
 * addToWriteWaiters - we have filled up an outgoing socket, and must
 *	stop sending to it for a while. Add that socket to the set
 *	we select on for output ready.
 */

void
addToWriteWaiters(GenericClient_t *gcl)
{

    FD_SET(gcl->gcl_socket, &ClientWriteBits);
}

/*
 * removeFromWriteWaiters - this client no longer needs to be woken up
 *	when something can be sent on its socket.
 */

void
removeFromWriteWaiters(GenericClient_t *gcl)
{

    FD_CLR(gcl->gcl_socket, &ClientWriteBits);
}

/*
 * scanClients - scan the client sockets for something we can do.
 */

static void
scanClients(fd_set *readFds, fd_set *writeFds, fd_set *exceptFds)
{
    GenericClient_t *gcl, *gclTemp;

    gcl = Clients;
    while (gcl != NULL) {
	if (FD_ISSET(gcl->gcl_socket, exceptFds)) {
	    gclTemp = gcl;
	    gcl = gcl->gcl_next;
	    abortClient(gclTemp);
	} else {
	    if (gcl->gcl_socket != INVALID_SOCKET &&
		FD_ISSET(gcl->gcl_socket, readFds))
	    {
		/* Something to read from the client socket. */
		(*gcl->gcl_type->clt_netClient->nc_read)
		    ((void *) gcl->gcl_clientData);
	    }
	    if (gcl->gcl_socket != INVALID_SOCKET &&
		FD_ISSET(gcl->gcl_socket, writeFds))
	    {
		/* We can write more to the client socket. */
		(*gcl->gcl_type->clt_netClient->nc_write)
		    ((void *) gcl->gcl_clientData);
	    }
	    gcl = gcl->gcl_next;
	}
    }
}

/*
 * handleConnection - a new connection has come in on a main socket.
 */

static void
handleConnection(ClientType_t *clt)
{
    struct sockaddr_in from;
    GenericClient_t *gcl;
    Socket_t s;
    long int flag;
#if defined(AMIGA)
    LONG fromLen;
#elif defined(SOLARIS) || defined(AIX)
    int fromLen;
#else
    unsigned int fromLen;
#endif

    DEBUG(("handleConnection from port %d\n", clt->clt_port));
    fromLen = sizeof(struct sockaddr_in);
    s = accept(clt->clt_socket, (struct sockaddr *) &from, &fromLen);
#if defined(_WIN32)
    if (s == INVALID_SOCKET) {
	s = WSAGetLastError();
	if (s != WSAEINTR && s != WSAEWOULDBLOCK) {
	    myPerror("accept");
#if DO_ABORTS
	    myAbort("client socket error");
#else
	    return;
#endif
	}
    }
#else
    if (s < 0) {
	if (errno != EINTR && errno != EWOULDBLOCK) {
	    myPerror("accept");
#if DO_ABORTS
	    myAbort("client socket error");
#else
	    return;
#endif
	}
    }
#endif
    else {
	/* Set the new client socket to be non-blocking. */
	flag = 1;
#if defined(_WIN32)
	if (ioctlsocket(s, FIONBIO, (unsigned long *) &flag) == SOCKET_ERROR) {
	    myPerror("ioctlsocket on new client socket");
	    killSocket(s);
	    return;
	}
#else
	if (ioctl(s, FIONBIO, (char *) &flag) == -1) {
	    myPerror("ioctl on new client socket");
	    killSocket(s);
	    return;
	}
#endif

	/* If the socket is a local one, then turn off the "Nagle
	   Algorithm" on it. That improves performance (a lot!) for
	   the local socket. */
	if (from.sin_addr.s_addr == MyAddr &&
	    setsockopt(s, IPPROTO_TCP, TCP_NODELAY, (char *) &flag,
		       sizeof(int)) != 0)
	{
	    myPerror("setsockopt on new socket");
	    killSocket(s);
	    return;
	}

	gcl = malloc(sizeof(GenericClient_t));
	if (gcl == NULL) {
	    killSocket(s);
	    return;
	}
	gcl->gcl_type = clt;
	gcl->gcl_socket = s;
	gcl->gcl_isShutDown = FALSE;
	gcl->gcl_clientData =
	    (*clt->clt_netClient->nc_alloc)(gcl, &TheNetGeneric,
					    clt->clt_auxData);
	if (gcl->gcl_clientData == NULL) {
	    killSocket(s);
	    free(gcl);
	    return;
	}

	/* All setup. Add it to our list of clients. */
	gcl->gcl_next = Clients;
	gcl->gcl_prev = NULL;
	if (Clients != NULL) {
	    Clients->gcl_prev = gcl;
	}
	Clients = gcl;

	/* Add this new socket to the set we 'select' on, awaiting input
	   from the user at the other end of it. */
	FD_SET(s, &AllSocketBits);
	if (s > MaxFd) {
	    MaxFd = s;
	}
    }
}

/*
 * netStart - wait for and handle requests from clients. This is a
 *	"mainline" routine - it does not return until it is asked to,
 *	*and* there are no clients left.
 */

bool_t
netStart(void)
{
    ClientType_t *clt;
    fd_set readFds, writeFds, exceptFds;
    struct timeval timeval;
    int res;
    bool_t mainException;

    while (! QuitRequested || Clients != NULL) {
	/* Modify this timeval if this code is triggering events. */
	timeval.tv_sec = 10;
	timeval.tv_usec = 0;
	readFds = AllSocketBits;
	writeFds = ClientWriteBits;
	exceptFds = readFds;
	DEBUG(("Going into select, MaxFd = %d, timeval = (%ld,%ld)\n",
	       MaxFd, timeval.tv_sec, timeval.tv_usec));

	res = select(MaxFd + 1, &readFds, &writeFds, &exceptFds, &timeval);

	DEBUG(("Got out of select, res = %d\n", res));
#if defined(_WIN32)
	if (res == SOCKET_ERROR)
#else
	if (res < 0)
#endif
	{
#if defined(_WIN32)
	    if (WSAGetLastError() == WSAEINTR)
#else
	    if (errno == EINTR)
#endif
	    {
		/* The select was interrupted. Assume it was an external
		   request for us to go away. */
		break;
	    } else {
		myPerror("select");
		break;
	    }
	} else if (res > 0) {

	    /* Activity on some socket. */
	    mainException = FALSE;
	    clt = ClientTypes;
	    while (clt != NULL) {
		if (FD_ISSET(clt->clt_socket, &exceptFds)) {
		    mainException = TRUE;
		    break;
		}
		clt = clt->clt_next;
	    }
	    if (mainException) {
		/* An exception on one of our main sockets. Bail out. */
		/* Better might be to try to shutdown that client type, but
		   that is harder if it has active clients. */
		break;
	    }

	    /* Check for a new connection on a main socket. */
	    clt = ClientTypes;
	    while (clt != NULL) {
		if (FD_ISSET(clt->clt_socket, &readFds)) {
		    handleConnection(clt);
		}
		clt = clt->clt_next;
	    }

	    /* Check for activity on the client sockets. */
	    scanClients(&readFds, &writeFds, &exceptFds);

	} else {
	    /* Nothing on any socket - timer has gone off. */
	    /* Put code here to handle timed events. */
	}
    }
    return TRUE;
}

/*
 * freeRequest - we provide this to client type code. It would normally
 *	call some real routine somewhere in the rest of the system.
 */

static void
freeRequest(Request_t *rq)
{
    free(rq);
}

#if defined(_WIN32)
#define MAXHOSTNAMELEN	INTERNET_MAX_HOST_NAME_LENGTH
#endif

/*
 * netInit - initial setup code for this module.
 */

bool_t
netInit(void)
{
    char nameBuffer[MAXHOSTNAMELEN];
    struct hostent *hp;

#if defined(_WIN32)
    if (WSAStartup(MAKEWORD(1, 0), &WSAData) < 0) {
	/* WinSock startup error. */
	return FALSE;
    }
#endif

    /* Basic initialization */
    ClientTypes = NULL;
    FD_ZERO(&AllSocketBits);
    FD_ZERO(&ClientWriteBits);
    MainSocketMax = 0;
    MaxFd = 0;
    NextId = 1;
    QuitRequested = FALSE;

    /* Find out the IP address of this machine. */
    if (gethostname(&nameBuffer[0], MAXHOSTNAMELEN) < 0) {
	myPerror("gethostname");
	return FALSE;
    }
    hp = gethostbyname(&nameBuffer[0]);
    if (hp == NULL) {
	myPerror("gethostbyname for self");
	return FALSE;
    }
#if defined(AMIGA)
    MyAddr = (long int) hp->h_addr;
#else
    MyAddr = (int) hp->h_addr;
#endif

    /* Setup the NetGeneric_t that we pass to client type code. */
    TheNetGeneric.ng_netWrite = netWrite;
    TheNetGeneric.ng_addToWriteWaiters = addToWriteWaiters;
    TheNetGeneric.ng_removeFromWriteWaiters = removeFromWriteWaiters;
    TheNetGeneric.ng_setWindowSize = NULL;
    TheNetGeneric.ng_setTerminalType = NULL;
    TheNetGeneric.ng_netRead = netRead;
    TheNetGeneric.ng_freeRequest = freeRequest;

    return TRUE;
}

/*
 * netSetBlank - outer code wants to change whether or not user input
 *	is currently being blanked (e.g. password blanking). Vector
 *	off to client-type specific code.
 */

void
netSetBlank(GenericClient_t *gcl, bool_t blankOn)
{
    if (gcl->gcl_type->clt_netClient->nc_setBlank != NULL) {
	(*gcl->gcl_type->clt_netClient->nc_setBlank)
	    ((void *) gcl->gcl_clientData, blankOn);
    }
}

/*
 * netSetPrompt - vector off to client-type specific code to set the
 *	current prompt.
 */

void
netSetPrompt(GenericClient_t *gcl, const char *newPrompt)
{
    if (gcl->gcl_type->clt_netClient->nc_setPrompt != NULL) {
	(*gcl->gcl_type->clt_netClient->nc_setPrompt)
	    ((void *) gcl->gcl_clientData, newPrompt);
    }
}

/*
 * netNewName - vector off to any client-type specific code that cares
 *	about the user changing names. In AmigaMUD, a binary client
 *	used the name in any logging it did.
 */

void
netNewName(GenericClient_t *gcl, const char *newName)
{
    if (gcl->gcl_type->clt_netClient->nc_newName != NULL) {
	(*gcl->gcl_type->clt_netClient->nc_newName)
	    ((void *) gcl->gcl_clientData, newName);
    }
}

/*
 * netPutText - required entry point. This is called by external code
 *	to send a Request_t full of text to the indicated client. We
 *	vector off to the appropriate code. A binary client might just
 *	send it directly, but a telnet client will have to filter the
 *	text through the telnet protocol.
 */

void
netPutText(GenericClient_t *gcl, Request_t *rq)
{
    /* There had better be one of these! */
    (*gcl->gcl_type->clt_netClient->nc_putText)
	((void *) gcl->gcl_clientData, rq);
}

/*
 * netFlush - vector off to a client-type specific routine to flush
 *	out any buffers it may be keeping for this client. This kind
 *	of buffering can reduce the number of system calls that we do,
 *	and also reduce the number of network packets we send.
 */

void
netFlush(GenericClient_t *gcl)
{
    if (gcl->gcl_type->clt_netClient->nc_flush != NULL) {
	(*gcl->gcl_type->clt_netClient->nc_flush)
	    ((void *) gcl->gcl_clientData);
    }
}

/*
 * netStop - external code wants us to go away. Note that we are normally
 *	sitting inside 'netStart', and will only leave that routine after
 *	being asked to (here), and when we have no clients left.
 */

void
netStop(void)
{

    QuitRequested = TRUE;
}

/*
 * netTerm - our actual termination routine. This should not be called
 *	until any calls to 'netStart' have returned.
 */

void
netTerm(void)
{
    GenericClient_t *gcl, *gclTemp;
    ClientType_t *clt, *cltTemp;

    gcl = Clients;
    while (gcl != NULL) {
	gclTemp = gcl;
	gcl = gcl->gcl_next;
	abortClient(gclTemp);
    }

    clt = ClientTypes;
    while (clt != NULL) {
	cltTemp = clt;
	clt = clt->clt_next;
	killSocket(cltTemp->clt_socket);
	free(cltTemp);
    }

#if defined(_WIN32)
    (void) WSACleanup();
#endif
}
