/*
 * telnet.c - code for the telnet client type.
 *	- Chris Gray, cg@ami-cg.GraySage.Edmonton.AB.CA, Nov/Dec 1998.
 */

/*
 * This code is provided as-is. No warranty of any kind is given.
 */

/*
 * The first bunch of code, in the 'DEBUG' #if, can safely be ignored.
 * It is only there to allow the remote user to examine and modify the
 * operating parameters of this code. If you plan on only allowing
 * telnet connections with specific, pre-setup, operating parameters,
 * you can just set 'DEBUG' to FALSE, and that will all go away.
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "types.h"
#include "telnetFlags.h"
#include "netClient.h"

/* Set DEBUG to FALSE to turn off '**' "commands", and debugging code */

#define DEBUG			TRUE

/* If DEBUG is TRUE and DEBUG_TO_USER is true, then output from debug
   and ** commands goes to the remote user. If DEBUG_TO_USER is false,
   it appears on stdout of the server. */

#define DEBUG_TO_USER		TRUE

/* Some parameters for various buffer sizes */

#define INPUT_BUFFER_SIZE	1024
#define LINE_BUFFER_SIZE	128
#define OUTPUT_BUFFER_SIZE	1024
#define SB_BUFFER_SIZE		512
#define DEBUG_BUFFER_SIZE	1024

/* Telnet protocol commands. Commands must start with IAC, followed by
   the command byte. Other occurrences of command bytes (not after IAC)
   are passed through. Options consist of IAC, one of the five
   negotiation bytes (DONT, DO, WONT, WILL, SB), then the option code. */

#define TELNET_IAC	255	/* Interprete As Command */
#define TELNET_DONT	254	/* Don't do option */
#define TELNET_DO	253	/* Do option */
#define TELNET_WONT	252	/* Won't do option */
#define TELNET_WILL	251	/* Will do option */
#define TELNET_SB	250	/* Subnegotiation begin */
#define TELNET_GA	249	/* Go Ahead */
#define TELNET_EL	248	/* Erase Line */
#define TELNET_EC	247	/* Erase Character */
#define TELNET_AYT	246	/* Are You There? */
#define TELNET_AO	245	/* Abort Output */
#define TELNET_IP	244	/* Interrupt Process */
#define TELNET_BRK	243	/* BREAK */
#define TELNET_DM	242	/* Data Mark */
#define TELNET_NOP	241	/* No Operation */
#define TELNET_SE	240	/* End of Subnegotiation parameters */
#define TELNET_EOR	239	/* End Of Record */
#define TELNET_ABORT	238
#define TELNET_SUSP	237	/* Suspend */
#define TELNET_EOF	236	/* End Of File */

/* Telnet option codes. Note that this is not a complete list! Also
   note that this code implements a very small number of these! */

#define BINARY		0	/* transmit binary (full 8 bits) */
#define ECHO		1	/* enable echoing from remote end */
#define SUPGA		3	/* SUPress Go Ahead */
#define STATUS		5	/* query status of options on other end */
#define TIMING_MARK	6	/* for flushing of pre-typed input, etc. */
#define NAOCRD		10	/* fiddling with carriage returns */
#define NAOHTS		11	/* fiddling with horizontal tabs */
#define NAOHTD		12	/* fiddling with horizontal tabs */
#define NAOFFD		13	/* fiddling with formfeeds */
#define NAOVTS		14	/* fiddling with vertical tabs */
#define NAOVTD		15	/* fiddling with vertical tabs */
#define NAOLFD		16	/* fiddling with linefeeds */
#define XTNDASC		17	/* extended ASCII option */
#define LOGOUT		18	/* forcible shutdown of user process */
#define BYTEMACRO	19	/* can support byte macros */
#define DET		20	/* Data Entry Terminal */
#define SUPDUP		21	/* a whole other protocol */
#define SUPDUPOUT	22	/* more of that other protocol */
#define SENDLOC		23	/* sending location of terminal */
#define TERMINALTYPE	24	/* Can send terminal type. */
#define ENDOFRECORD	25	/* send end-of-record indicators */
#define OUTMRK		27	/* about sending banners to user */
#define TTYLOC		28	/* CMU 64 bit identification numbers */
#define MODE3270	29	/* remember those IBM 3270 terminals? */
#define X3PAD		30	/* enable X.3 protocol stuff */
#define NAWS		31	/* Negotiate About Window Size */
#define TTYSPEED	32	/* tell server what speed tty is */
#define FLOWCNTL	33	/* pass flow control over the link */
#define LINEMODE	34	/* the *big* complex real line-mode */
#define XDISPLOC	35	/* send X display name to server */
#define ENVIRON		36	/* username, etc. */
#define AUTH		37	/* authentication of the ends */
#define NEWENVIRON	39	/* obsoletes ENVIRON */
#define MAX_OPTION	39	/* for handling with arrays */
#define EXOPL		255	/* Extended option negotiation */

/* These are the characters this code senses as line-editing characters.
   Sorry, they aren't configurable. Proper LINEMODE support would do that.
   ^R redraws input, ^U and ^X erase the current input. */

#define CNTL_R		0x12
#define CNTL_U		0x15
#define CNTL_X		0x18

/* The structure of the per-client data we use to support telnet. */

typedef struct TelnetClient {
    const struct GenericClient *tncl_client;	/* net code's data */
    const NetGeneric_t *tncl_netGeneric;	/* net interface */
    Request_t *tncl_outGoing;			/* outgoing queue of data */
    unsigned int tncl_outGoingPos;		/* position in head element */
    unsigned int tncl_outputPos;		/* next free in outputBuf */
    unsigned int tncl_outputCurrent;		/* next to send in "" */
    unsigned int tncl_flags;			/* various option bits */
    unsigned int tncl_linePos;			/* input line position */
    unsigned int tncl_SBPos;			/* telnet subnegotiation */
#if DEBUG
    unsigned int tncl_debugPos;			/* next free in debug buf */
    char tncl_debugBuffer[DEBUG_BUFFER_SIZE + 2];
    char tncl_utilBuffer[20];			/* for formatting stuff */
    bool_t tncl_debugCmds;			/* debug telnet negotiation */
    bool_t tncl_debugPrompt;			/* debug prompt happenings */
    bool_t tncl_debugInput;			/* show received lines */
#endif
    bool_t tncl_echoOn;				/* not blanking */
    bool_t tncl_supGA;				/* suppres GoAhead? */
    bool_t tncl_doEOR;				/* send ENDOFRECORDs? */
    byte_t tncl_inputBuffer[INPUT_BUFFER_SIZE];	/* raw input bytes */
    byte_t tncl_SBBuffer[SB_BUFFER_SIZE];	/* telet subnegotiation */
    char tncl_lineBuffer[LINE_BUFFER_SIZE + 1];	/* cooked input bytes */
    byte_t tncl_outputBuffer[OUTPUT_BUFFER_SIZE];/* cooked output bytes */
    char *tncl_prompt;				/* current prompt */
    char *tncl_erase;				/* for erasing prompt */
    unsigned int tncl_promptLen;		/* so can write them */
    bool_t tncl_sentOptionDO[MAX_OPTION + 1];	/* used in negotiation */
    bool_t tncl_sentOptionWILL[MAX_OPTION + 1];	/* "" */
    bool_t tncl_sentOptionDONT[MAX_OPTION + 1];	/* "" */
    bool_t tncl_sentOptionWONT[MAX_OPTION + 1];	/* "" */
    bool_t tncl_promptShown;			/* is it shown now? */
    bool_t tncl_lastWasCR;			/* handle CR or CR/LF */
    bool_t tncl_meDoEcho;			/* am I echoing to user? */
    bool_t tncl_sendBinary;			/* negotiation result */
    bool_t tncl_receiveBinary;			/* negotiation result */
    enum {
	tntst_normal,
	tntst_command,
	tntst_SB
    } tncl_telnetState;				/* how to process new bytes */
    bool_t tncl_lastWasIAC;			/* in telnet command now */
    byte_t tncl_command;			/* which command is it? */
    bool_t tncl_writeWaiting;			/* I'm queued to write */
    bool_t tncl_doneOutput;			/* something to send out? */
} TelnetClient_t;

/* These macros are used in various places to produce debug output. That
   output is under control of some user-settable flags if DEBUG is TRUE. */

#if DEBUG

#define CMD_DEBUG(tncl, m) \
    if (tncl->tncl_debugCmds) { \
	debug(tncl, m); \
    }
#define CMD_DEBUG_CMD(tncl, m, cmd, byte)   debugCmd(tncl, m, cmd, byte)
#define CMD_DEBUG_SHORT_CMD(tncl, m, cmd)   debugShortCmd(tncl, m, cmd)
#define DEBUG_PROMPT(tncl, m)		    debugPrompt(tncl, m)

#else

#define CMD_DEBUG(tncl, m)
#define CMD_DEBUG_CMD(tncl, m, cmd, byte)
#define CMD_DEBUG_SHORT_CMD(tncl, m, cmd)
#define DEBUG_PROMPT(tncl, m)

#endif

/* Keep a pointer to the routine to handle input lines. It is passed the
   GenericClient pointer so that it can identify the client. */

static LineHandler_t LineHandler;

#if DEBUG

/* I don't know the names of them all. Look up more RFC's! */

static char *(OptionNames[]) = {
    "BINARY", "ECHO", "2", "SUPGA", "4", "STATUS", "TIMING_MARK",
    "7", "8", "9", "NAOCRD", "NAOHTS", "NAOHTD", "NAOFFD", "NAOVTS",
    "NAOVTD", "NAOLFD", "XTNDASC", "LOGOUT", "BYTEMACRO", "DET",
    "SUPDUP", "SUPDUPOUT", "SENDLOC", "TERMINALTYPE", "ENDOFRECORD",
    "26", "OUTMRK", "TTYLOC", "MODE3270", "X3PAD", "NAWS", "TTYSPEED",
    "FLOWCNTL", "LINEMODE", "XDISPLOC", "ENVIRON", "AUTH", "38",
    "NEWENVIRON"
};

/* I believe this set to be complete */

static char *(CmdNames[]) = {
    "EOF", "SUSP", "ABORT", "EOR", "SE", "NOP", "DM", "BRK", "IP", "AO",
    "AYT", "EC", "EL", "GA", "SB", "WILL", "WONT", "DO", "DONT", "IAC"
};

/* Forward declarations. */

static void sendChars(TelnetClient_t *tncl, char *p, unsigned int len);
static void debugString(TelnetClient_t *tncl, char *m);

/*
 * Still within the #if DEBUG, we now have some low-level stuff to handle
 * a buffer of debug output maintained per client.
 */

/*
 * debugEnd - end of a debugging sequence - flush the buffer.
 */

static void
debugEnd(TelnetClient_t *tncl)
{

    if (tncl->tncl_debugPos != 0) {
	tncl->tncl_debugBuffer[tncl->tncl_debugPos] = '\n';
#if DEBUG_TO_USER
	sendChars(tncl, &tncl->tncl_debugBuffer[0], tncl->tncl_debugPos + 1);
#else
	/* Yes, there is space! */
	tncl->tncl_debugBuffer[tncl->tncl_debugPos + 1] = '\0';
	fprintf(stderr, "%s", &tncl->tncl_debugBuffer[0]);
#endif
    }
}

/*
 * debugChar - add a character to the debug buffer.
 */

static void
debugChar(TelnetClient_t *tncl, char ch)
{

    if (tncl->tncl_debugPos == DEBUG_BUFFER_SIZE) {
	debugEnd(tncl);
	debugString(tncl, "**+ ");
    }
    tncl->tncl_debugBuffer[tncl->tncl_debugPos] = ch;
    tncl->tncl_debugPos += 1;
}

/*
 * debugString - add a string to the debug output.
 */

static void
debugString(TelnetClient_t *tncl, char *m)
{

    while (*m != '\0') {
	debugChar(tncl, *m);
	++m;
    }
}

/*
 * debugStart - first step in a debug sequence. Add a prefix.
 */

static void
debugStart(TelnetClient_t *tncl)
{

    tncl->tncl_debugPos = 0;
    debugString(tncl, "** ");
}

/*
 * debugByte - add the decimal form of a byte to the debug output.
 */

static void
debugByte(TelnetClient_t *tncl, byte_t b)
{

    if (b >= 100) {
	debugChar(tncl, (byte_t) (b / 100 + '0'));
	b %= 100;
    }
    if (b >= 10) {
	debugChar(tncl, (byte_t) (b / 10 + '0'));
	b %= 10;
    }
    debugChar(tncl, (byte_t) (b + '0'));
}

/*
 * debug - used during debugging - send a string as a separate debug line.
 */

static void
debug(TelnetClient_t *tncl, char *message)
{

    debugStart(tncl);
    debugString(tncl, message);
    debugEnd(tncl);
}

/*
 * debugWithBuffer - common for promptDebug and doInputLine.
 */

static void
debugWithBuffer(TelnetClient_t *tncl, char *m, char *buf)
{

    debugStart(tncl);
    debugString(tncl, m);
    debugString(tncl, ": ");
    debugString(tncl, buf);
    debugEnd(tncl);
}

/*
 * fmtByte - utility for optionName and cmdName. Just format the byte
 *	in decimal in the per-client utility buffer.
 */

static char *
fmtByte(TelnetClient_t *tncl, byte_t b)
{
    char *p;

    p = &tncl->tncl_utilBuffer[0];
    if (b >= 100) {
	*p++ = b / 100 + '0';
	b %= 100;
    }
    if (b >= 10) {
	*p++ = b / 10 + '0';
	b %= 10;
    }
    *p++ = b % 10 + '0';
    *p = '\0';
    return &tncl->tncl_utilBuffer[0];
}

/*
 * optionName - return the name for a telnet option.
 */

static char *
optionName(TelnetClient_t *tncl, byte_t option)
{

    if (option <= NEWENVIRON) {
	return OptionNames[option];
    } else if (option == EXOPL) {
	return "EXOPL";
    } else {
	return fmtByte(tncl, option);
    }
}

/*
 * cmdName - return the name for a telnet short command.
 */

static char *
cmdName(TelnetClient_t *tncl, byte_t cmd)
{

    if (cmd >= TELNET_EOF) {
	return CmdNames[cmd - TELNET_EOF];
    } else {
	return fmtByte(tncl, cmd);
    }
}

/*
 * debugCmd - used during debugging. Display a message about a telnet
 *	command being sent or received.
 */

static void
debugCmd(TelnetClient_t *tncl, char *m, byte_t cmd, byte_t option)
{

    if (tncl->tncl_debugCmds) {
	debugStart(tncl);
	debugString(tncl, m);
	debugChar(tncl, ' ');
	switch (cmd) {
	case TELNET_DONT:
	    debugString(tncl, "DONT");
	    break;
	case TELNET_DO:
	    debugString(tncl, "DO");
	    break;
	case TELNET_WONT:
	    debugString(tncl, "WONT");
	    break;
	case TELNET_WILL:
	    debugString(tncl, "WILL");
	    break;
	default:
	    debugString(tncl, "???");
	    break;
	}
	debugChar(tncl, ' ');
	debugString(tncl, optionName(tncl, option));
	debugEnd(tncl);
    }
}

/*
 * debugShortCmd - ditto.
 */

static void
debugShortCmd(TelnetClient_t *tncl, char *m, byte_t cmd)
{

    if (tncl->tncl_debugCmds) {
	debugStart(tncl);
	debugString(tncl, m);
	debugChar(tncl, ' ');
	debugString(tncl, cmdName(tncl, cmd));
	debugEnd(tncl);
    }
}

/*
 * debugPrompt - used during debugging of prompt operations.
 */

static void
debugPrompt(TelnetClient_t *tncl, char *message)
{

    if (tncl->tncl_debugPrompt) {
	debugWithBuffer(tncl, message, tncl->tncl_prompt);
    }
}

#endif

/*
 * OK, now we some of the lower-level routines for outputting stuff to
 * the socket. They are used by normal text output and by the interactive
 * commands.
 */

/*
 * pushBuffer - try to make progress with sending from the client buffer
 *	out over the socket. Return TRUE if there is room in the buffer.
 */

static bool_t
pushBuffer(TelnetClient_t *tncl)
{
    unsigned int current;
    long int wantLen, sentLen;

    current = tncl->tncl_outputCurrent;
    wantLen = tncl->tncl_outputPos - current;
    if (wantLen == 0) {
	return TRUE;
    }
    sentLen = (tncl->tncl_netGeneric->ng_netWrite)
	((struct GenericClient *) tncl->tncl_client,
	 &tncl->tncl_outputBuffer[current], wantLen);
    if (sentLen > 0) {
	if (sentLen == wantLen) {
	    /* Buffer is now empty. */
	    tncl->tncl_outputCurrent = 0;
	    tncl->tncl_outputPos = 0;
	} else {
	    /* Made progress, but did not empty the buffer. */
	    tncl->tncl_outputCurrent = current + sentLen;
	}
	return TRUE;
    }
    return FALSE;
}

/*
 * putByte - bottom level interface for sending data to client. If we are
 *	stuck, because our buffer is full, then return FALSE, and the
 *	stall notification must propagate all the way back up.
 *	(Well, we do this for the main output of text from the rest of
 *	the system, but we are not so careful with prompts and debug stuff.)
 */

static bool_t
putByte(TelnetClient_t *tncl, byte_t b)
{
    unsigned int pos, current;

    /* We allow a few bytes of slop, so that we can reasonably handle the
       cases where we have to escape a character or send commands. */
    pos = tncl->tncl_outputPos;
    if (pos >= OUTPUT_BUFFER_SIZE - 5) {
	/* Reached end of output buffer - can we shift it down? */
	current = tncl->tncl_outputCurrent;
	if (current != 0) {
	    /* Yes, that makes space. */
	    memcpy(&tncl->tncl_outputBuffer[0],
		   &tncl->tncl_outputBuffer[current], pos - current);
	    pos -= current;
	    tncl->tncl_outputPos = pos;
	    tncl->tncl_outputCurrent = 0;
	} else {
	    /* No, have to try to flush to player. */
	    if (! tncl->tncl_writeWaiting) {
		if (! pushBuffer(tncl)) {
		    /* We are stuck. Request notification of socket unblock. */
		    (*tncl->tncl_netGeneric->ng_addToWriteWaiters)
			((struct GenericClient *) tncl->tncl_client);
		    tncl->tncl_writeWaiting = TRUE;
		}
	    }
	}
    }
    pos = tncl->tncl_outputPos;
    if (pos < OUTPUT_BUFFER_SIZE) {
	/* Lose any bytes we have no room for. This should only happen
	   in awkward places like prompt changes, when the socket is
	   filled up. I could fix this with a couple more variables
	   in the TelnetClient structure, marking where in the prompt
	   or erase I am, and handle those first in 'telnetWrite'. */
	tncl->tncl_outputBuffer[pos] = b;
	pos += 1;
	tncl->tncl_outputPos = pos;
	tncl->tncl_doneOutput = TRUE;
    }
    return (bool_t) ! tncl->tncl_writeWaiting;
}

/*
 * sendShortCmd - send one of the one-byte commands. No need for a call
 *	to 'pushBuffer' here, since these are only sent in a stream of
 *	other stuff, never by themselves.
 */

static void
sendShortCmd(TelnetClient_t *tncl, byte_t cmd)
{

    (void) putByte(tncl, TELNET_IAC);
    (void) putByte(tncl, cmd);
    CMD_DEBUG_SHORT_CMD(tncl, "send", cmd);
}

/*
 * sendCmd - send a telnet command, pushing it out the socket.
 */

static void
sendCmd(TelnetClient_t *tncl, byte_t cmd, byte_t option)
{

    (void) putByte(tncl, TELNET_IAC);
    (void) putByte(tncl, cmd);
    (void) putByte(tncl, option);
    CMD_DEBUG_CMD(tncl, "send", cmd, option);
    (void) pushBuffer(tncl);
}

/*
 * sendOptionDO - request that a given option be enabled.
 */

static void
sendOptionDO(TelnetClient_t *tncl, byte_t option)
{

    sendCmd(tncl, TELNET_DO, option);
    tncl->tncl_sentOptionDO[option] = TRUE;
}

/*
 * sendOptionWILL - offer to enable an option.
 */

static void
sendOptionWILL(TelnetClient_t *tncl, byte_t option)
{

    sendCmd(tncl, TELNET_WILL, option);
    tncl->tncl_sentOptionWILL[option] = TRUE;
}

/*
 * sendOptionDONT - tell the other guy to not do something.
 */

static void
sendOptionDONT(TelnetClient_t *tncl, byte_t option)
{

    sendCmd(tncl, TELNET_DONT, option);
    tncl->tncl_sentOptionDONT[option] = TRUE;
}

/*
 * sendOptionWONT - refuse to do something.
 */

static void
sendOptionWONT(TelnetClient_t *tncl, byte_t option)
{

    sendCmd(tncl, TELNET_WONT, option);
    tncl->tncl_sentOptionWONT[option] = TRUE;
}

/*
 * From here for a while, is the code to handle the '**' interactive
 * commands that let the remote user examine and control things. All of
 * this can be removed along with controllable debug output, by setting
 * 'DEBUG' to 'FALSE' at the top of the file.
 */

#if DEBUG

/*
 * debugFlag - helper for doDebugCommand.
 */

static void
debugFlag(TelnetClient_t *tncl, char *name, bool_t value)
{

    debugStart(tncl);
    debugString(tncl, name);
    debugString(tncl, ": ");
    debugString(tncl, value ? "TRUE" : "FALSE");
    debugEnd(tncl);
}

/*
 * debugBit - another helper for doDebugCommand
 */

static void
debugBit(TelnetClient_t *tncl, char *name, unsigned int bit)
{

    debugStart(tncl);
    debugString(tncl, name);
    debugString(tncl, ": ");
    debugString(tncl, (tncl->tncl_flags & bit) != 0 ? "TRUE" : "FALSE");
    debugEnd(tncl);
}

/*
 * debugToggleFlag - another helper for doDebugCommand
 */

static void
debugToggleFlag(TelnetClient_t *tncl, char *name, bool_t *pFlag)
{

    debugStart(tncl);
    debugString(tncl, name);
    debugString(tncl, " display turned ");
    if (*pFlag) {
	debugString(tncl, "off");
	*pFlag = FALSE;
    } else {
	debugString(tncl, "on");
	*pFlag = TRUE;
    }
    debugEnd(tncl);
}

/*
 * debugToggleBit - another helper for doDebugCommand
 */

static void
debugToggleBit(TelnetClient_t *tncl, char *name, unsigned int bit)
{

    debugStart(tncl);
    debugString(tncl, name);
    debugString(tncl, " turned ");
    if ((tncl->tncl_flags & bit) != 0) {
	debugString(tncl, "off");
	tncl->tncl_flags &= ~bit;
    } else {
	debugString(tncl, "on");
	tncl->tncl_flags |= bit;
    }
    debugEnd(tncl);
}

/*
 * doDebugCommand - remote user has issued a debug command.
 */

static void
doDebugCommand(TelnetClient_t *tncl)
{
    char *p;

    p = &tncl->tncl_lineBuffer[2];
    tncl->tncl_lineBuffer[tncl->tncl_linePos] = '\0';
    if (strcmp(p, "?") == 0) {
	debug(tncl, "Telnet mode commands:\n\n"
	      "**? - show this info\n"
	      "**st - show current server telnet status\n"
	      "**cmd - toggle telnet option negotiation display\n"
	      "**prompt - toggle prompt activity display\n"
	      "**input - toggle input line display\n"
	      "**ga - toggle any sending of GA\n"
	      "**pwblank - toggle attempts to blank passwords\n"
	      "**pwsupga - toggle use of SUPGA in password blanking\n"
	      "**eraseprompt - toggle attempts to erase prompts\n"
	      "**binary - renegotiate binary\n"
	      "**echo - renegotiate echo\n"
	      "**eor - renegotiate ENDOFRECORD\n"
	      "**naws - force NAWS redo\n"
	      "**ttytype - force TERMINALTYPE redo\n"
	      "**fbinary - force binary on/off\n"
	      "**fecho - force server echo on/off");
    } else if (strcmp(p, "st") == 0) {
	debugFlag(tncl, "sendBinary", tncl->tncl_sendBinary);
	debugFlag(tncl, "receiveBinary", tncl->tncl_receiveBinary);
	debugFlag(tncl, "server echo", tncl->tncl_meDoEcho);
	debugFlag(tncl, "suppress goahead", tncl->tncl_supGA);
	debugFlag(tncl, "use end of record", tncl->tncl_doEOR);
#if 0
	/* These values (the window size and the terminal type) are no
	   longer maintained directly by this code. Instead, they are
	   passed to higher level code whenever the values are
	   received. So, we cannot display them here. */
	debugStart(tncl);
	debugString(tncl, "terminal type is '");
	debugString(tncl, tncl->tncl_terminalType);
	debugChar(tncl, '\'');
	debugEnd(tncl);
	debugStart(tncl);
	debugString(tncl, "display height ");
	debugByte(tncl, tncl->tncl_client->cl_textHeight);
	debugString(tncl, " width ");
	debugByte(tncl, tncl->tncl_client->cl_textWidth);
	debugEnd(tncl);
#endif
	debugBit(tncl, "ever send GA", B_SEND_GA);
	debugBit(tncl, "attempt password blanking", B_PW_BLANK);
	debugBit(tncl, "use SUPGA on password blanking", B_PW_BLANK_SUPGA);
	debugBit(tncl, "erase prompts", B_ERASE_PROMPT);
    } else if (strcmp(p, "cmd") == 0) {
	debugToggleFlag(tncl, "telnet option negotiation",
			&tncl->tncl_debugCmds);
    } else if (strcmp(p, "prompt") == 0) {
	debugToggleFlag(tncl, "prompt activity", &tncl->tncl_debugPrompt);
    } else if (strcmp(p, "input") == 0) {
	debugToggleFlag(tncl, "input line", &tncl->tncl_debugInput);
    } else if (strcmp(p, "ga") == 0) {
	debugToggleBit(tncl, "send of GA's", B_SEND_GA);
    } else if (strcmp(p, "pwblank") == 0) {
	debugToggleBit(tncl, "blanking of passwords", B_PW_BLANK);
    } else if (strcmp(p, "pwsupga") == 0) {
	debugToggleBit(tncl, "use of SUPGA in password blanking",
		       B_PW_BLANK_SUPGA);
    } else if (strcmp(p, "eraseprompt") == 0) {
	debugToggleBit(tncl, "attempts to erase prompt", B_ERASE_PROMPT);
    } else if (strcmp(p, "binary") == 0) {
	debug(tncl, "attempting to enable binary data");
	sendOptionWILL(tncl, BINARY);
	sendOptionDO(tncl, BINARY);
    } else if (strcmp(p, "echo") == 0) {
	if (tncl->tncl_meDoEcho) {
	    debug(tncl, "asking client to echo");
	    sendOptionWONT(tncl, ECHO);
	    if ((tncl->tncl_flags & B_PW_BLANK_SUPGA) != 0) {
		debug(tncl, "asking for no SUPGA");
		sendOptionWONT(tncl, SUPGA);
		sendOptionDONT(tncl, SUPGA);
	    }
	} else {
	    debug(tncl, "offering to echo");
	    sendOptionWILL(tncl, ECHO);
	    if ((tncl->tncl_flags & B_PW_BLANK_SUPGA) != 0) {
		debug(tncl, "asking for SUPGA");
		sendOptionWILL(tncl, SUPGA);
		sendOptionDO(tncl, SUPGA);
	    }
	}
    } else if (strcmp(p, "eor") == 0) {
	if (tncl->tncl_doEOR) {
	    debug(tncl, "trying to turn EOR off");
	    sendOptionWONT(tncl, ENDOFRECORD);
	    sendOptionDONT(tncl, ENDOFRECORD);
	} else {
	    debug(tncl, "trying to turn EOR on");
	    sendOptionWILL(tncl, ENDOFRECORD);
	    sendOptionDO(tncl, ENDOFRECORD);
	}
    } else if (strcmp(p, "naws") == 0) {
	debug(tncl, "re-requesting NAWS");
	sendOptionDO(tncl, NAWS);
    } else if (strcmp(p, "ttytype") == 0) {
	debug(tncl, "re-requesting TTYTYPE");
	sendOptionDO(tncl, TERMINALTYPE);
    } else if (strcmp(p, "fbinary") == 0) {
	if (tncl->tncl_sendBinary) {
	    debug(tncl, "forcing BINARY off");
	    tncl->tncl_sendBinary = FALSE;
	} else {
	    debug(tncl, "forcing BINARY on");
	    tncl->tncl_sendBinary = TRUE;
	}
	tncl->tncl_receiveBinary = tncl->tncl_sendBinary;
    } else if (strcmp(p, "fecho") == 0) {
	if (tncl->tncl_meDoEcho) {
	    debug(tncl, "server will no longer echo");
	    tncl->tncl_meDoEcho = FALSE;
	} else {
	    debug(tncl, "server will start echoing");
	    tncl->tncl_meDoEcho = TRUE;
	}
    } else {
	debug(tncl, "Invalid telnet mode command. **? for help");
    }
}

#endif

/*
 * Now we have some lower-level stuff that is actually involved with
 * sending text to the user through the telnet protocol.
 */

/*
 * sendChar - bottom level of putting characters to the client. This
 *	handles any translation needed on the character for the telnet
 *	protocol rules.
 */

static bool_t
sendChar(TelnetClient_t *tncl, char ch)
{

    if ((unsigned char) ch == TELNET_IAC) {
	/* Yes - single '&' - want both executed. */
	return (putByte(tncl, TELNET_IAC) & putByte(tncl, TELNET_IAC));
    }
    if ((ch & 0x80) != 0 && ! tncl->tncl_sendBinary) {
	ch &= 0x7f;
    }
    if (ch == '\n') {
	return (putByte(tncl, '\r') & putByte(tncl, '\n'));
    }
    return putByte(tncl, ch);
}

/*
 * sendChars - send a known-length array of characters. Ignore stalling.
 */

static void
sendChars(TelnetClient_t *tncl, char *p, unsigned int len)
{

    while (len != 0) {
	(void) sendChar(tncl, *p);
	++p;
	len -= 1;
    }
}

/*
 * sendString - send a string. Ignore stalling.
 */

static void
sendString(TelnetClient_t *tncl, char *st)
{

    while (*st != '\0') {
	(void) sendChar(tncl, *st);
	++st;
    }
}

/*
 * complain - a complaint from the telnet code to the remote user.
 */

static void
complain(TelnetClient_t *tncl, char *message)
{

    sendString(tncl, "** ");
    sendString(tncl, message);
    sendChar(tncl, '\n');
}

/*
 * showPrompt - show the current prompt.
 */

static void
showPrompt(TelnetClient_t *tncl, bool_t doGA)
{

    sendChars(tncl, tncl->tncl_prompt, tncl->tncl_promptLen);
    tncl->tncl_promptShown = TRUE;
    if (doGA) {
	if (tncl->tncl_doEOR) {
	    sendShortCmd(tncl, TELNET_EOR);
	} else if (! tncl->tncl_supGA && (tncl->tncl_flags & B_SEND_GA) != 0) {
	    sendShortCmd(tncl, TELNET_GA);
	}
    }
}

/*
 * unshowPrompt - remove the current prompt.
 */

static void
unshowPrompt(TelnetClient_t *tncl)
{

    sendChars(tncl, tncl->tncl_erase, tncl->tncl_promptLen + 4);
    tncl->tncl_promptShown = FALSE;
}

/*
 * processOutgoing - a server text request. Process the text
 *	of the message through the telnet protocol, buffering it in
 *	the client buffer. Then, try sending the data. If we managed
 *	to send all of it, or there is still some room in the output
 *	buffer, return TRUE, else return FALSE.
 */

static bool_t
processOutgoing(TelnetClient_t *tncl, Request_t *rq, unsigned int startPos)
{
    char *p;
    unsigned int i;

    if (startPos == 0 && tncl->tncl_promptShown &&
	(tncl->tncl_flags & B_ERASE_PROMPT) != 0)
    {
	DEBUG_PROMPT(tncl, "processOutgoing unshowing");
	unshowPrompt(tncl);
    }

    p = &rq->rq_u.ru_text[startPos];
    for (i = startPos; i != rq->rq_usedLen; i += 1) {
	if (! sendChar(tncl, *p)) {
	    /* We have plugged up. Must pause here. */
	    tncl->tncl_outGoingPos = i;
	    return FALSE;
	}
	++p;
    }

    /* All buffered up. If the buffer is nearly full, try sending it
       right away. This is a heuristic to try to keep the user seeing
       full output messages, rather than partial ones. */
    if (tncl->tncl_outputPos >= OUTPUT_BUFFER_SIZE - 40) {
	return pushBuffer(tncl);
    }
    return TRUE;
}

/*
 * telnetWrite - called when we can write to this client socket.
 */

static void
telnetWrite(void *clientData)
{
    TelnetClient_t *tncl;
    Request_t *rq, *rqTemp;
    bool_t newLineAtEnd;

    tncl = (TelnetClient_t *) clientData;
    /* Is there something in the output buffer? */
    if (tncl->tncl_outputCurrent != tncl->tncl_outputPos) {
	/* Yes, try to push it out. */
	if (! pushBuffer(tncl)) {
	    return;
	}
    }

    newLineAtEnd = FALSE;
    /* Is there a partially sent message sitting around? */
    rq = tncl->tncl_outGoing;
    if (rq != NULL && tncl->tncl_outGoingPos != 0) {
	/* Yes, try to process the remainder. */
	if (! processOutgoing(tncl, rq, tncl->tncl_outGoingPos)) {
	    return;
	}
	rqTemp = rq;
	rq = rq->rq_next;
	if (rqTemp->rq_u.ru_text[rqTemp->rq_usedLen - 1] == '\n') {
	    newLineAtEnd = TRUE;
	}
	(*tncl->tncl_netGeneric->ng_freeRequest)(rqTemp);
    }

    /* OK, try any other queued up messages. */
    while (rq != NULL) {
	if (! processOutgoing(tncl, rq, 0)) {
	    break;
	}
	rqTemp = rq;
	rq = rq->rq_next;
	if (rqTemp->rq_u.ru_text[rqTemp->rq_usedLen - 1] == '\n') {
	    newLineAtEnd = TRUE;
	}
	(*tncl->tncl_netGeneric->ng_freeRequest)(rqTemp);
    }
    tncl->tncl_outGoing = rq;
    if (rq == NULL) {
	if (newLineAtEnd) {
	    /* The end of the last outgoing text request was a newline. This
	       is a good place to show the prompt. */
	    DEBUG_PROMPT(tncl, "telnetWrite showing");
	    showPrompt(tncl, TRUE);
	}
	(*tncl->tncl_netGeneric->ng_removeFromWriteWaiters)
	    ((struct GenericClient *) tncl->tncl_client);
    }
}

/*
 * telnetSetPrompt - set in a new prompt.
 */

static void
telnetSetPrompt(void *clientData, const char *prompt)
{
    TelnetClient_t *tncl;
    char *newPrompt, *newErase;
    unsigned int len;
    bool_t promptWasShown;

    tncl = (TelnetClient_t *) clientData;
    promptWasShown = tncl->tncl_promptShown;
    if (promptWasShown) {
	if ((tncl->tncl_flags & B_ERASE_PROMPT) != 0) {
	    DEBUG_PROMPT(tncl, "telnetSetPrompt unshowing");
	    unshowPrompt(tncl);
	} else {
	    sendChar(tncl, '\n');
	}
    }
    len = strlen(prompt);
    newErase = malloc(len + 4);
    newPrompt = malloc(len + 1);
    if (newErase != NULL && newPrompt != NULL) {
	free(tncl->tncl_erase);
	free(tncl->tncl_prompt);
	tncl->tncl_erase = newPrompt;
	newErase[0] = '\r';
	newErase[1] = '\0';
	strcpy(&newErase[2], prompt);
	strcat(&newErase[2], "\r");
	tncl->tncl_prompt = newPrompt;
	strcpy(newPrompt, prompt);
	tncl->tncl_promptLen = len;
    } else {
	if (newErase != NULL) {
	    free(newErase);
	}
	if (newPrompt != NULL) {
	    free(newPrompt);
	}
    }
    if (promptWasShown) {
	DEBUG_PROMPT(tncl, "telnetSetPrompt showing");
	showPrompt(tncl, TRUE);
    }
    if (! tncl->tncl_writeWaiting) {
	(void) pushBuffer(tncl);
    }
}

/*
 * telnetSetBlank - control blanking, for password entering.
 */

static void
telnetSetBlank(void *clientData, bool_t doBlank)
{
    TelnetClient_t *tncl;

    tncl = (TelnetClient_t *) clientData;
    /* Seems backwards but it is the other end's echoing we are
       controlling. It is "conventional" to link ECHO with SUPGA, so doing
       so here will hopefully allow us to work properly with more TELNET
       clients. */
    if (! doBlank) {
	if (tncl->tncl_meDoEcho && (tncl->tncl_flags & B_PW_BLANK) != 0) {
	    sendOptionWONT(tncl, ECHO);
	    if ((tncl->tncl_flags & B_PW_BLANK_SUPGA) != 0) {
		sendOptionWONT(tncl, SUPGA);
		sendOptionDONT(tncl, SUPGA);
	    }
	}
	CMD_DEBUG(tncl, "setting EchoOn");
	tncl->tncl_echoOn = TRUE;
    } else {
	if (! tncl->tncl_meDoEcho && (tncl->tncl_flags & B_PW_BLANK) != 0) {
	    sendOptionWILL(tncl, ECHO);
	    if ((tncl->tncl_flags & B_PW_BLANK_SUPGA) != 0) {
		sendOptionWILL(tncl, SUPGA);
		sendOptionDO(tncl, SUPGA);
	    }
	}
	CMD_DEBUG(tncl, "clearing EchoOn");
	tncl->tncl_echoOn = FALSE;
    }
}

/*
 * doInputLine - stuff for sending a line for further processing.
 */

static void
doInputLine(TelnetClient_t *tncl)
{

    if (tncl->tncl_meDoEcho) {
	(void) sendChar(tncl, '\n');
    } else {
	(void) sendChar(tncl, '\r');
    }
#if DEBUG
    if (tncl->tncl_linePos >= 3 && tncl->tncl_lineBuffer[0] == '*' &&
	tncl->tncl_lineBuffer[1] == '*')
    {
	doDebugCommand(tncl);
	tncl->tncl_linePos = 0;
	return;
    }
    if (tncl->tncl_debugInput) {
	/* lineBuffer is one longer than we ever put into it */
	tncl->tncl_lineBuffer[tncl->tncl_linePos] = '\0';
	debugWithBuffer(tncl, "input", &tncl->tncl_lineBuffer[0]);
    }
#endif
    tncl->tncl_promptShown = FALSE;

    (*LineHandler)(tncl->tncl_client,
		   &tncl->tncl_lineBuffer[0], tncl->tncl_linePos);

    tncl->tncl_linePos = 0;
}

/*
 * telnetStopClient - normal or abrupt shutdown of a telnet client.
 *	If 'doShutDown' is true, we should call out to any scenario
 *	(world) code that should be done when a client goes away. This
 *	could include things like messages to other players in the room,
 *	saying that 'such-and-so has left the game'.
 */

static void
telnetStopClient(void *clientData, bool_t doShutDown)
{
}

/*
 * Regular text input-line processing stuff here.
 */

/*
 * doBackup - erase a character.
 */

static void
doBackup(TelnetClient_t *tncl)
{

    (void) sendChar(tncl, '\b');
    (void) sendChar(tncl, ' ');
    (void) sendChar(tncl, '\b');
}

/*
 * cookChar - handle a single input character. Return TRUE if the
 *	character was a carriage return (or LF), and we have just called
 *	doInputLine to process the user input, else return FALSE.
 */

static bool_t
cookChar(TelnetClient_t *tncl, char ch)
{
    unsigned int i;

    switch (ch) {
    case '\r':
	doInputLine(tncl);
	tncl->tncl_lastWasCR = TRUE;
	return TRUE;
    case '\n':
	if (! tncl->tncl_lastWasCR) {
	    doInputLine(tncl);
	    return TRUE;
	}
	break;
    case '\b':
    case 0x7f:
	if (tncl->tncl_linePos != 0) {
	    tncl->tncl_linePos -= 1;
	    if (tncl->tncl_meDoEcho) {
		doBackup(tncl);
	    }
	}
	break;
    case CNTL_X:
    case CNTL_U:
	if (tncl->tncl_linePos != 0) {
	    if (tncl->tncl_meDoEcho) {
		for (i = 0; i != tncl->tncl_linePos; i += 1) {
		    doBackup(tncl);
		}
	    }
	    tncl->tncl_linePos = 0;
	}
	break;
    case CNTL_R:
	if (tncl->tncl_meDoEcho && tncl->tncl_linePos != 0) {
	    sendChar(tncl, '^');
	    sendChar(tncl, 'R');
	    sendChar(tncl, '\n');
	    DEBUG_PROMPT(tncl, "CNTL_R showing");
	    showPrompt(tncl, FALSE);
	    for (i = 0; i != tncl->tncl_linePos; i += 1) {
		sendChar(tncl, tncl->tncl_lineBuffer[i]);
	    }
	}
	break;
    default:
	if (tncl->tncl_linePos != LINE_BUFFER_SIZE) {
	    tncl->tncl_lineBuffer[tncl->tncl_linePos] = ch;
	    tncl->tncl_linePos += 1;
	    if (tncl->tncl_meDoEcho) {
		if (tncl->tncl_echoOn) {
		    sendChar(tncl, ch);
		} else {
		    sendChar(tncl, ' ');
		}
	    }
	}
	break;
    }
    tncl->tncl_lastWasCR = FALSE;
    return FALSE;
}

/*
 * Finally we get into the real telnet protocol handling.
 */

/*
 * processSB - process a subnegotiation sequence in tncl_SBBuffer.
 */

static void
processSB(TelnetClient_t *tncl)
{
    unsigned int i, j, len;
    byte_t *p;

#if DEBUG
    if (tncl->tncl_debugCmds) {
	debugStart(tncl);
	debugString(tncl, "SB ");
	if (tncl->tncl_SBPos == 0) {
	    debugString(tncl, "len = 0");
	} else {
	    debugString(tncl, optionName(tncl, tncl->tncl_SBBuffer[0]));
	    if (tncl->tncl_SBPos != 1) {
		for (i = 1; i < tncl->tncl_SBPos; i += 1) {
		    debugChar(tncl, ' ');
		    debugByte(tncl, tncl->tncl_SBBuffer[i]);
		}
	    }
	}
	debugEnd(tncl);
    }
#endif

    if (tncl->tncl_SBPos != 0) {
	switch (tncl->tncl_SBBuffer[0]) {
	case NAWS:
	    /* RFC 1073 */
	    if (tncl->tncl_SBPos == 5) {
		/* telnet protocol does everything big-endian */
		i = (unsigned int) tncl->tncl_SBBuffer[1] << 8 |
		    (unsigned int) tncl->tncl_SBBuffer[2];
		j = (unsigned int) tncl->tncl_SBBuffer[3] << 8 |
		    (unsigned int) tncl->tncl_SBBuffer[4];
		if (i == 0) {
		    i = 80;
		}
		if (j == 0) {
		    j = 24;
		}
#if DEBUG
		if (tncl->tncl_debugCmds) {
		    debugStart(tncl);
		    debugString(tncl, "NAWS got height ");
		    debugByte(tncl, (byte_t) j);
		    debugString(tncl, ", width ");
		    debugByte(tncl, (byte_t) i);
		    debugEnd(tncl);
		}
#endif
		if (tncl->tncl_netGeneric->ng_setWindowSize != NULL) {
		    (*tncl->tncl_netGeneric->ng_setWindowSize)
			((struct GenericClient *) tncl->tncl_client, j, i);
		}
	    } else {
		complain(tncl, "invalid NAWS SB received");
	    }
	    break;
	case TERMINALTYPE:
	    /* RFC 1091 */
	    if (tncl->tncl_SBPos >= 4 && tncl->tncl_SBBuffer[1] == 0) {
		p = &tncl->tncl_SBBuffer[2];
		len = tncl->tncl_SBPos - 2;
		/* TERMINAL-TYPE IS ... IAC */
		while (*p != TELNET_IAC && len != 0) {
		    ++p;
		    len -= 1;
		}
		*p = '\0';
		if (tncl->tncl_netGeneric->ng_setTerminalType != NULL) {
		    (*tncl->tncl_netGeneric->ng_setTerminalType)
			((struct GenericClient *) tncl->tncl_client,
			 (char *) &tncl->tncl_SBBuffer[2]);
		}
#if DEBUG
		if (tncl->tncl_debugCmds) {
		    debugStart(tncl);
		    debugString(tncl, "Got terminal type '");
		    debugString(tncl, (char *) &tncl->tncl_SBBuffer[2]);
		    debugChar(tncl, '\'');
		    debugEnd(tncl);
		}
#endif
	    } else {
		complain(tncl, "invalid TERMINALTYPE SB received");
	    }
	    break;
	default:
	    complain(tncl, "unknown SB command received");
	}
    }
}

/*
 * cookByte - first level of interpreting data from user. This
 *	level handles all of the TELNET protocol stuff.
 */

static bool_t
cookByte(TelnetClient_t *tncl, byte_t b)
{
    bool_t sentLine;

    sentLine = FALSE;
    switch (tncl->tncl_telnetState) {
    case tntst_normal:
	if (tncl->tncl_lastWasIAC) {
	    switch (b) {
	    case TELNET_IAC:
		/* Should only get these if other end is in binary, i.e. if
		   "ReceiveBinary" is set. */
		CMD_DEBUG_SHORT_CMD(tncl, "recv", TELNET_IAC);
		(void) cookChar(tncl, (char) b);
		tncl->tncl_telnetState = tntst_normal;
		break;
	    case TELNET_AYT:
		/* Yep, I'm here. */
		CMD_DEBUG_SHORT_CMD(tncl, "recv", TELNET_AYT);
		sendShortCmd(tncl, TELNET_NOP);
		tncl->tncl_telnetState = tntst_normal;
		break;
	    case TELNET_GA:
	    case TELNET_AO:
	    case TELNET_DM:
	    case TELNET_NOP:
	    case TELNET_EOR:
	    case TELNET_SUSP:
		/* Just ignore these. */
		CMD_DEBUG_SHORT_CMD(tncl, "recv", b);
		tncl->tncl_telnetState = tntst_normal;
		break;
	    case TELNET_DONT:
	    case TELNET_DO:
	    case TELNET_WONT:
	    case TELNET_WILL:
		tncl->tncl_command = b;
		tncl->tncl_telnetState = tntst_command;
		break;
	    case TELNET_SB:
		tncl->tncl_telnetState = tntst_SB;
		tncl->tncl_SBPos = 0;
		break;
	    case TELNET_SE:
		complain(tncl, "invalid SE without matching SB");
		tncl->tncl_telnetState = tntst_normal;
		break;
	    case TELNET_EC:
		CMD_DEBUG_SHORT_CMD(tncl, "recv", TELNET_EC);
		(void) cookChar(tncl, '\b');
		tncl->tncl_telnetState = tntst_normal;
		break;
	    case TELNET_EL:
		CMD_DEBUG_SHORT_CMD(tncl, "recv", TELNET_EL);
		(void) cookChar(tncl, CNTL_X);
		tncl->tncl_telnetState = tntst_normal;
		break;
	    case TELNET_IP:
	    case TELNET_BRK:
	    case TELNET_ABORT:
	    case TELNET_EOF:
		/* Do it gently */
		CMD_DEBUG_SHORT_CMD(tncl, "recv", b);
		telnetStopClient((void *) tncl->tncl_client, TRUE);
		break;
	    }
	    tncl->tncl_lastWasIAC = FALSE;
	} else {
	    if (b == TELNET_IAC) {
		tncl->tncl_lastWasIAC = TRUE;
	    } else if (b != 0) {
		sentLine = cookChar(tncl, (char) b);
	    }
	}
	break;
    case tntst_command:
	tncl->tncl_telnetState = tntst_normal;
	CMD_DEBUG_CMD(tncl, "recv", tncl->tncl_command, b);
	switch (tncl->tncl_command) {
	case TELNET_DO:
	    if (b > MAX_OPTION) {
		/* Just ignore it. */
		break;
	    }
	    if (tncl->tncl_sentOptionWILL[b] || tncl->tncl_sentOptionWONT[b]) {
		tncl->tncl_sentOptionWILL[b] = FALSE;
		tncl->tncl_sentOptionWONT[b] = FALSE;
		switch (b) {
		case BINARY:
		    CMD_DEBUG(tncl, "setting SendBinary");
		    tncl->tncl_sendBinary = TRUE;
		    break;
		case ECHO:
		    CMD_DEBUG(tncl, "setting MeDoEcho");
		    tncl->tncl_meDoEcho = TRUE;
		    break;
		case SUPGA:
		    CMD_DEBUG(tncl, "setting SupGA");
		    tncl->tncl_supGA = TRUE;
		    break;
		case ENDOFRECORD:
		    CMD_DEBUG(tncl, "setting DoEOR");
		    tncl->tncl_doEOR = TRUE;
		}
	    } else {
		switch (b) {
		case BINARY:
		    CMD_DEBUG(tncl, "setting SendBinary");
		    tncl->tncl_sendBinary = TRUE;
		    sendCmd(tncl, TELNET_WILL, b);
		    break;
		case ECHO:
		    CMD_DEBUG(tncl, "setting MeDoEcho");
		    tncl->tncl_meDoEcho = TRUE;
		    sendCmd(tncl, TELNET_WILL, b);
		    break;
		case SUPGA:
		    CMD_DEBUG(tncl, "setting SupGA");
		    tncl->tncl_supGA = TRUE;
		    sendCmd(tncl, TELNET_WILL, b);
		    break;
		case ENDOFRECORD:
		    CMD_DEBUG(tncl, "setting DoEOR");
		    tncl->tncl_doEOR = TRUE;
		    break;
		case NAWS:
		    sendCmd(tncl, TELNET_WILL, b);
		    break;
		default:
		    /* No, I won't do it! */
		    sendCmd(tncl, TELNET_WONT, b);
		    break;
		}
	    }
	    break;
	case TELNET_WILL:
	    if (b > MAX_OPTION) {
		break;
	    }
	    if (tncl->tncl_sentOptionDO[b] || tncl->tncl_sentOptionDONT[b]) {
		tncl->tncl_sentOptionDO[b] = FALSE;
		tncl->tncl_sentOptionDONT[b] = FALSE;
		switch (b) {
		case BINARY:
		    CMD_DEBUG(tncl, "setting ReceiveBinary");
		    tncl->tncl_receiveBinary = TRUE;
		    break;
		case ECHO:
		    CMD_DEBUG(tncl, "clearing MeDoEcho");
		    tncl->tncl_meDoEcho = FALSE;
		    break;
		case SUPGA:
		case NAWS:
		    break;
		case TERMINALTYPE:
		    CMD_DEBUG(tncl, "asking for terminal type");
		    putByte(tncl, TELNET_IAC);
		    putByte(tncl, TELNET_SB);
		    putByte(tncl, TERMINALTYPE);
		    putByte(tncl, 1);	/* SEND = 1 */
		    putByte(tncl, TELNET_IAC);
		    putByte(tncl, TELNET_SE);
		    break;
		default:
		    sendCmd(tncl, TELNET_DONT, b);
		    break;
		}
	    } else {
		switch (b) {
		case BINARY:
		    CMD_DEBUG(tncl, "setting ReceiveBinary");
		    tncl->tncl_receiveBinary = TRUE;
		    sendCmd(tncl, TELNET_DO, b);
		    break;
		case ECHO:
		    CMD_DEBUG(tncl, "clearing MeDoEcho");
		    tncl->tncl_meDoEcho = FALSE;
		    sendCmd(tncl, TELNET_DO, b);
		    break;
		case SUPGA:
		case NAWS:
		case TERMINALTYPE:
		    /* Sure, do these - I understand them. */
		    sendCmd(tncl, TELNET_DO, b);
		    break;
		default:
		    /* Don't need to send a DONT, since that is default,
		       but do it in case someone needs it. */
		    sendCmd(tncl, TELNET_DONT, b);
		    break;
		}
	    }
	    break;
	case TELNET_DONT:
	    if (b > MAX_OPTION) {
		break;
	    }
	    if (tncl->tncl_sentOptionWONT[b] || tncl->tncl_sentOptionWILL[b]) {
		tncl->tncl_sentOptionWONT[b] = FALSE;
		tncl->tncl_sentOptionWILL[b] = FALSE;
		switch (b) {
		case BINARY:
		    CMD_DEBUG(tncl, "clearing SendBinary");
		    tncl->tncl_sendBinary = FALSE;
		    break;
		case ECHO:
		    CMD_DEBUG(tncl, "clearing MeDoEcho");
		    tncl->tncl_meDoEcho = FALSE;
		    break;
		case SUPGA:
		    CMD_DEBUG(tncl, "clearing SupGA");
		    tncl->tncl_supGA = FALSE;
		    break;
		case ENDOFRECORD:
		    CMD_DEBUG(tncl, "clearing DoEOR");
		    tncl->tncl_doEOR = FALSE;
		    break;
		}
	    } else {
		switch (b) {
		case BINARY:
		    CMD_DEBUG(tncl, "clearing SendBinary");
		    tncl->tncl_sendBinary = FALSE;
		    break;
		case ECHO:
		    CMD_DEBUG(tncl, "clearing MeDoEcho");
		    tncl->tncl_meDoEcho = FALSE;
		    break;
		case SUPGA:
		    CMD_DEBUG(tncl, "clearing SupGA");
		    tncl->tncl_supGA = FALSE;
		    break;
		case ENDOFRECORD:
		    CMD_DEBUG(tncl, "clearing DoEOR");
		    tncl->tncl_doEOR = FALSE;
		    break;
		}
		sendCmd(tncl, TELNET_WONT, b);
	    }
	    break;
	case TELNET_WONT:
	    if (b > MAX_OPTION) {
		break;
	    }
	    if (tncl->tncl_sentOptionDO[b] || tncl->tncl_sentOptionDONT[b]) {
		tncl->tncl_sentOptionDO[b] = FALSE;
		tncl->tncl_sentOptionDONT[b] = FALSE;
		switch (b) {
		case BINARY:
		    CMD_DEBUG(tncl, "clearing ReceiveBinary");
		    tncl->tncl_receiveBinary = FALSE;
		    break;
		case ECHO:
		    CMD_DEBUG(tncl, "setting MeDoEcho");
		    tncl->tncl_meDoEcho = TRUE;
		    break;
		}
	    } else {
		switch (b) {
		case BINARY:
		    CMD_DEBUG(tncl, "clearing ReceiveBinary");
		    tncl->tncl_receiveBinary = FALSE;
		    break;
		case ECHO:
		    CMD_DEBUG(tncl, "setting MeDoEcho");
		    tncl->tncl_meDoEcho = TRUE;
		    break;
		}
		sendCmd(tncl, TELNET_DONT, b);
	    }
	}
	break;
    case tntst_SB:
	if (tncl->tncl_lastWasIAC) {
	    if (b == TELNET_IAC) {
		if (tncl->tncl_SBPos < SB_BUFFER_SIZE) {
		    tncl->tncl_SBBuffer[tncl->tncl_SBPos] = b;
		    tncl->tncl_SBPos += 1;
		}
	    } else if (b == TELNET_SE) {
		tncl->tncl_telnetState = tntst_normal;
		processSB(tncl);
	    } else {
		complain(tncl, "invalid IAC in SB");
		tncl->tncl_telnetState = tntst_normal;
	    }
	    tncl->tncl_lastWasIAC = FALSE;
	} else if (b == TELNET_IAC) {
	    tncl->tncl_lastWasIAC = TRUE;
	} else {
	    if (tncl->tncl_SBPos < SB_BUFFER_SIZE) {
		tncl->tncl_SBBuffer[tncl->tncl_SBPos] = b;
		tncl->tncl_SBPos += 1;
	    }
	}
	break;
    }
    return sentLine;
}

/*
 * OK, that's done. Now some stuff that is glue between this code and
 * other code, such as the networking code.
 */

/*
 * telnetRead - called when there is something to read on this client.
 */

static void
telnetRead(void *clientData)
{
    TelnetClient_t *tncl;
    char *p;
    long int len;
    bool_t sentLine;

    tncl = (TelnetClient_t *) clientData;
    sentLine = FALSE;
    tncl->tncl_doneOutput = FALSE;
    p = (char *) &tncl->tncl_inputBuffer[0];
    len = (*tncl->tncl_netGeneric->ng_netRead)
	((struct GenericClient *) tncl->tncl_client, p, INPUT_BUFFER_SIZE);
    sentLine = FALSE;
    while (len > 0) {
	if (cookByte(tncl, *p)) {
	    sentLine = TRUE;
	}
	len -= 1;
	++p;
    }
    if (sentLine && ! tncl->tncl_promptShown) {
	DEBUG_PROMPT(tncl, "telnetRead showing");
	showPrompt(tncl, TRUE);
    }
    if (tncl->tncl_doneOutput && ! tncl->tncl_writeWaiting) {
	(void) pushBuffer(tncl);
    }
}

/*
 * telnetAllocClient - a new connection has arrived - setup our
 *	client structure for it, returning a pointer to it.
 */

static const void *
telnetAllocClient(struct GenericClient *gcl, const NetGeneric_t *ng, int flags)
{
    TelnetClient_t *tncl;
    unsigned int i;

    tncl = malloc(sizeof(TelnetClient_t));
    if (tncl == NULL) {
	return NULL;
    }

    tncl->tncl_client = gcl;
    tncl->tncl_netGeneric = ng;
    tncl->tncl_flags = flags;
    tncl->tncl_outGoing = NULL;
    tncl->tncl_outGoingPos = 0;
    tncl->tncl_outputPos = 0;
    tncl->tncl_outputCurrent = 0;
    tncl->tncl_linePos = 0;
#if DEBUG
    tncl->tncl_debugPos = 0;
    tncl->tncl_debugCmds = FALSE;
    tncl->tncl_debugPrompt = FALSE;
    tncl->tncl_debugInput = FALSE;
#endif

    tncl->tncl_promptLen = 0;
    tncl->tncl_erase = strdup("\rX\r");
    tncl->tncl_erase[1] = '\0';
    tncl->tncl_prompt = strdup("");
    if (tncl->tncl_erase == NULL || tncl->tncl_prompt == NULL) {
	if (tncl->tncl_erase != NULL) {
	    free(tncl->tncl_erase);
	}
	if (tncl->tncl_prompt != NULL) {
	    free(tncl->tncl_prompt);
	}
	free(tncl);
	return NULL;
    }
    for (i = 0; i != MAX_OPTION; i += 1) {
	tncl->tncl_sentOptionDO[i] = FALSE;
	tncl->tncl_sentOptionWILL[i] = FALSE;
	tncl->tncl_sentOptionDONT[i] = FALSE;
	tncl->tncl_sentOptionWONT[i] = FALSE;
    }
    tncl->tncl_echoOn = TRUE;
    tncl->tncl_promptShown = TRUE;	/* forces the first one out */
    tncl->tncl_lastWasCR = FALSE;
    tncl->tncl_telnetState = tntst_normal;
    tncl->tncl_lastWasIAC = FALSE;
    tncl->tncl_writeWaiting = FALSE;

    /* Set these to all FALSE, as per telnet specs */
    tncl->tncl_sendBinary = FALSE;
    tncl->tncl_receiveBinary = FALSE;
    tncl->tncl_meDoEcho = FALSE;
    tncl->tncl_supGA = FALSE;
    tncl->tncl_doEOR = FALSE;

    if ((tncl->tncl_flags & B_NEG_BINARY) != 0) {
	sendOptionWILL(tncl, BINARY);
	sendOptionDO(tncl, BINARY);
    }
    if ((tncl->tncl_flags & B_NEG_PSEUDO) != 0) {
	sendOptionWONT(tncl, ECHO);
	sendOptionWONT(tncl, SUPGA);
    } else {
	if ((tncl->tncl_flags & B_NEG_ECHO) != 0) {
	    sendOptionWILL(tncl, ECHO);
	}
	if ((tncl->tncl_flags & B_NEG_SUPGA) != 0) {
	    sendOptionWILL(tncl, SUPGA);
	    sendOptionDO(tncl, SUPGA);
	}
    }
    if ((tncl->tncl_flags & B_NEG_TTYTYPE) != 0) {
	sendOptionDO(tncl, TERMINALTYPE);
    }
    if ((tncl->tncl_flags & B_NEG_EORMODE) != 0) {
	/* We will do EOR stuff, be we don't want it. */
	sendOptionWILL(tncl, ENDOFRECORD);
	sendOptionDONT(tncl, ENDOFRECORD);
    }
    if ((tncl->tncl_flags & B_NEG_NAWS) != 0) {
	sendOptionDO(tncl, NAWS);
    }

    /* Startup overrides of proper telnet protocol defaults */
    if ((tncl->tncl_flags & B_INIT_ECHO) != 0) {
	tncl->tncl_meDoEcho = TRUE;
    }
    if ((tncl->tncl_flags & B_INIT_EORMODE) != 0) {
	tncl->tncl_doEOR = TRUE;
    }

    telnetWrite(tncl);

    return (const void *) tncl;
}

/*
 * telnetFreeClient - the outside world is done with this client.
 */

static void
telnetFreeClient(void *clientData)
{
    TelnetClient_t *tncl;

    tncl = (TelnetClient_t *) clientData;
    free(tncl->tncl_erase);
    free(tncl->tncl_prompt);
    free(tncl);
}

/*
 * telnetPutText - interface to send and/or queue a text output request.
 */

static void
telnetPutText(void *clientData, Request_t *rq)
{
    TelnetClient_t *tncl;
    Request_t **pRq;

    if (rq->rq_usedLen == 0) {
	return;
    }
    tncl = (TelnetClient_t *) clientData;
    if (tncl->tncl_writeWaiting) {
	/* Already plugged up - just queue the new message. */
	pRq = &tncl->tncl_outGoing;
	while (*pRq != NULL) {
	    pRq = &(*pRq)->rq_next;
	}
	*pRq = rq;
    } else {
	/* Try it right away. */
	if (processOutgoing(tncl, rq, 0)) {
	    /* Sent/buffered the whole thing right away. */
	    if (rq->rq_u.ru_text[rq->rq_usedLen - 1] == '\n') {
		DEBUG_PROMPT(tncl, "telnetPutText showing");
		showPrompt(tncl, TRUE);
	    }
	    (*tncl->tncl_netGeneric->ng_freeRequest)(rq);
	} else {
	    /* Still some left - have to queue it. tncl_writeWaiting will
	       have been set. */
	    rq->rq_next = NULL;
	    tncl->tncl_outGoing = rq;
	}
    }
}

/*
 * telnetInit - initialize here, and return a pointer to a NetClient_t
 *	containing our routines. Note that we have no modifiable
 *	data here, so there is no setup required.
 */

NetClient_t *
telnetInit(LineHandler_t lineHandler)
{
    static NetClient_t nc;

    LineHandler = lineHandler;
    nc.nc_alloc = telnetAllocClient;
    nc.nc_setBlank = telnetSetBlank;
    nc.nc_setPrompt = telnetSetPrompt;
    nc.nc_newName = NULL;
    nc.nc_putText = telnetPutText;
    nc.nc_flush = telnetWrite;
    nc.nc_stop = telnetStopClient;
    nc.nc_read = telnetRead;
    nc.nc_write = telnetWrite;
    nc.nc_free = telnetFreeClient;
    return &nc;
}
