/* Example DevMUD module to exercise interface standard and socket module */

#ifndef WIN32
#include <sys/time.h>
#else
#include "decl.h"
#include <time.h>
#endif

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include "../include/devmud.h"

/* TODO static number of players is very bad to assume */
#define MAX_PLAYERS 200 /* only 256 descriptors... */
#define READ_BUFFER_SIZE 16384
#define OUT_BUFFER_SIZE (READ_BUFFER_SIZE+256)
#define LASTLOG_SIZE 10

static void (*close_connection)(int which);
static void (*parsed_put_string)(int to, char *message);
static int (*get_message)(char *buffer, int size, int sec, int usec);
static int (*real_connection)(int which);
static char *(*connection_name)(int which);
static int (*connection_time)(int which);

static char names[MAX_PLAYERS][100];

static char command_buffer[MAX_PLAYERS][READ_BUFFER_SIZE];
static int command_tail[MAX_PLAYERS];
static int state[MAX_PLAYERS];

static void say(char *str)
{
	int loop;

	for (loop=0;loop<MAX_PLAYERS;loop++)
	{
		if (real_connection(loop))
		{
			parsed_put_string(loop,str);
		}
	}
	/* For logging of the local discussion */
	fprintf(stderr,"%s",str);
}

static void who(int which)
{
	char output_buffer[OUT_BUFFER_SIZE];
	int loop;
	int curtime, idletime, delta;
	char *timename;

	curtime = time(0L);
	parsed_put_string(which,"Player -1 (logger)\r\n");
	for (loop=0;loop<MAX_PLAYERS;loop++)
	{
		if (real_connection(loop))
		{
			timename = "second";
			idletime = connection_time(loop);
			delta = curtime - idletime;
			if (delta > 60)
			{
				delta = delta/60;
				timename = "minute";
				if (delta > 60)
				{
					delta = delta/60;
					timename = "hour";
					if (delta > 12)
					{
						delta = delta/12;
						timename = "day";
						if (delta > 7)
						{
							delta = delta/7;
							timename = "week";
						}
					}
				}
			}
			if (delta != 0)
			{
				sprintf(output_buffer,
					"Player %d%s, idle %d %s%s\r\n",
					loop, names[loop], delta, timename,
					((delta == 1) ? "" : "s"));
			} else {
				sprintf(output_buffer,"Player %d%s\r\n",
					loop, names[loop]);
			}
			parsed_put_string(which,output_buffer);
		}
	}
}

static void parse_input(int which, char *buffer)
{
	char *command;
	char output_buffer[OUT_BUFFER_SIZE];

	/* read until newline, or... */
	while ((*buffer != '\n') && (*buffer != '\r') &&
	/* we really filled up the buffer. */
	       (command_tail[which] < READ_BUFFER_SIZE))
	{
		command_buffer[which][command_tail[which]] = *buffer;
		if (*buffer == 0)
		{
			return;
		}
		command_tail[which]++;
		buffer++;
	}
	/* terminate the string, if needed */
	command_buffer[which][command_tail[which]] = 0;
	/* and inform the next time that there isn't a partial buffer */
	command_tail[which] = 0;
	command = command_buffer[which];

	if (state[which] == 1)	/* chat mode */
	{
		if (strcmp(command,".") == 0)
		{
			state[which] = 0;
			sprintf(output_buffer, "Ending chat mode.\r\n");
			parsed_put_string(which, output_buffer);
		} else {
			sprintf(output_buffer,"\r\nPlayer %d%s says: %s\r\n", which, names[which], command);
			say(output_buffer);
		}
	}
	else if (strcmp(command,"chat") == 0)
	{
		state[which] = 1;	/* player is in chat mode */
		sprintf(output_buffer, "Now in chat mode.  End with \".\" alone on a line.\r\n");
		parsed_put_string(which, output_buffer);
	}
	else if (strcmp(command,"commandchat") == 0)
	{
		state[which] = 2;	/* player is in commandchat mode */
		sprintf(output_buffer, "Now in commandchat mode.  End with \".\" alone on a line.\r\n");
		parsed_put_string(which, output_buffer);
	}
	else if (strncmp(command,"say ",4) == 0)
	{
		sprintf(output_buffer,"\r\nPlayer %d%s says: %s\r\n", which, names[which], command+4);
		say(output_buffer);
	}
	else if (strncmp(command,"\"",1) == 0)
	{
		sprintf(output_buffer,"\r\nPlayer %d%s says: %s\r\n", which, names[which], command+1);
		say(output_buffer);
	}
	else if (strncmp(command,"'",1) == 0)
	{
		sprintf(output_buffer,"\r\nPlayer %d%s says: %s\r\n", which, names[which], command+1);
		say(output_buffer);
	}
	else if (strncmp(command,"emote ",6) == 0)
	{
		sprintf(output_buffer,"\r\nPlayer %d%s %s\r\n", which, names[which], command+6);
		say(output_buffer);
	}
	else if (strncmp(command,":",1) == 0)
	{
		sprintf(output_buffer,"\r\nPlayer %d%s %s\r\n", which, names[which], command+1);
		say(output_buffer);
	}
	else if (strncmp(command,";",1) == 0)
	{
		sprintf(output_buffer,"\r\nPlayer %d%s%s\r\n", which, names[which], command+1);
		say(output_buffer);
	}
/*
	else if (strncmp(command,"tell ",5) == 0)
	{
		sprintf(output_buffer,"\r\nPlayer %d%s%s\r\n", which, names[which], command+1);
		say(output_buffer);
	}
*/
	/* exact command matches */
	else if ((strcmp(command,"who") == 0) ||
	         (strcmp(command,"w") == 0) ||
	         (strcmp(command,"look") == 0) ||
	         (strcmp(command,"l") == 0))
	{
		who(which);
	}
	else if (strncmp(command,"name ",5) == 0)
	{
		if (strlen(command) < 90)
		{
			sprintf(names[which]," (%s)", command+5);
			sprintf(output_buffer, "Now Player %d%s\r\n", which, names[which]);
			parsed_put_string(which, output_buffer);
		}
	}
	else if (strncmp(command,"help",4) == 0)
	{
		parsed_put_string(which, "Commands are:\r\n");
		parsed_put_string(which, "\"chat\": say strings.\r\n");
		parsed_put_string(which, "\"commandchat\": say strings that aren't commands.\r\n");
		parsed_put_string(which, "\".\": end a chat mode.\r\n");
		parsed_put_string(which, "\"say \": say a string.\r\n");
		parsed_put_string(which, "\"\"\": say a string.\r\n");
		parsed_put_string(which, "\"'\": say a string.\r\n");
		parsed_put_string(which, "\"emote \": emote a string.\r\n");
		parsed_put_string(which, "\":\": emote a string.\r\n");
		parsed_put_string(which, "\";\": emote a string with no extra space.\r\n");
/*
		parsed_put_string(which, "\"tell \": tell another player a string.\r\n");
*/
		parsed_put_string(which, "\"help\": print commands.\r\n");
		parsed_put_string(which, "\"name \": set your name.\r\n");
		parsed_put_string(which, "\"who\": list connected players.\r\n");
		parsed_put_string(which, "\"w\": alias for who.\r\n");
		parsed_put_string(which, "\"look\": alias for who.\r\n");
		parsed_put_string(which, "\"l\": alias for who.\r\n");
		parsed_put_string(which, "\"quit\": leave the game.\r\n");
		parsed_put_string(which, "World version number is 10.\r\n");
	}
	else if (strcmp(command,".") == 0)
	{
		if (state[which] == 0)
		{
			sprintf(output_buffer, "Ending nonexistent chat mode.\r\n");
		} else {
			sprintf(output_buffer, "Ending chat mode.\r\n");
		}
		parsed_put_string(which, output_buffer);
		state[which] = 0;
	}
	/* exact command match */
	else if (strcmp(command,"quit") == 0)
	{
		sprintf(output_buffer,"\r\nPlayer %d%s quit.\r\n",which, names[which]);
		say(output_buffer);
		close_connection(which);
	}
	else /* unrecognized command case */
	{
		if (*command)
		{
			if (state[which] == 0)
			{
				sprintf(output_buffer,"Command \"%s\" unrecognized: try \"help\".\r\n",command);
				parsed_put_string(which, output_buffer);
			} else {
				sprintf(output_buffer,"\r\nPlayer %d%s says: %s\r\n", which, names[which], command);
				say(output_buffer);
			}
		}
	}
}

static void new_player(int which)
{
	char output_buffer[OUT_BUFFER_SIZE];

	names[which][0] = 0;
	state[which] = 0;
	command_tail[which] = 0;
	sprintf(output_buffer,"Player %d", which);
	sprintf(output_buffer,"Welcome, %d! (%s)\r\n", which, connection_name(which));
	say(output_buffer);
}

static void end_player(int which)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"Bye, %d!\r\n",which);
	say(output_buffer);
}

struct iface world_supplies[] = {
	{NULL,NULL,NULL,NULL}
};

struct iface world_uses[] = {
	{ "close_connection", NULL, "void(int)", NULL },
	{ "parsed_put_string", NULL, "void(int,char*)", NULL },
	{ "get_message", NULL, "int(char*,int,int,int)", NULL },
	{ "real_connection", NULL, "int(int)", NULL },
	{ "connection_name", NULL, "char*(int)", NULL },
	{ "connection_time", NULL, "int(int)", NULL },
	{ "config_port", NULL, "void(int)", NULL },
	{ NULL, NULL, NULL, NULL }
};

void world_use_functions(struct iface *it)
{
	struct iface *temp;

	temp = it;
	while (temp->name != NULL)
	{
		if (!strcmp(temp->name,"close_connection"))
		{
			close_connection = temp->function;
		}
		else if (!strcmp(temp->name,"parsed_put_string"))
		{
			parsed_put_string = temp->function;
		}
		else if (!strcmp(temp->name,"get_message"))
		{
			get_message = temp->function;
		}
		else if (!strcmp(temp->name,"real_connection"))
		{
			real_connection = temp->function;
		}
		else if (!strcmp(temp->name,"connection_name"))
		{
			connection_name = temp->function;
		}
		else if (!strcmp(temp->name,"connection_time"))
		{
			connection_time = temp->function;
		}
		temp++;
	}
	return;
}

int world_initialize(void)
{
	return(0);
}

int world_start(void)
{
	char read_buffer[READ_BUFFER_SIZE];
	int i, which;

	for(i=0;i<MAX_PLAYERS;i++)
	{
		command_buffer[i][0] = 0;
		command_tail[i] = 0;
		state[i] = 0;
	}
	while(1)
	{
		read_buffer[0] = 0;
		while ((which = get_message(read_buffer,READ_BUFFER_SIZE,1,0)) == -1)
			;
		if (*read_buffer)
		{
			parse_input(which, read_buffer);
		} else {
			if (real_connection(which))
				new_player(which);
			else
				end_player(which);
		}
	}
	return(0);
}

int world_stop(void)
{
	return(-1);
}
