#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef WIN32
#include <dlfcn.h>
#else
#include <windows.h>
#endif
#include "../include/devmud.h"

extern struct module_handle *static_modules[];

int main_argc;
char **main_argv;

static void *extract_symbol(struct module_handle *handle, char *name)
{
	void *sym;
	const char *error;
#ifndef WIN32
	sym = dlsym(handle->dl_handle, name);
	if ((error = dlerror()) != NULL) {
#else
	sym = GetProcAddress(handle->dl_handle, name);
	if (sym == NULL) {
#endif
		fprintf(stderr, "Could not load:%s\n", name);
		return(NULL);	/* DevMUD symbols may not be NULL */
	}
	return(sym);
}

/* a NULL return indicates failure to load */
struct module_handle *dl_load_module(char *name)
{
	void *dl_handle;
	struct module_handle *it;
	char *space;

	/* 15 is space for the _use_functions, and all the shorter strings. */
	space = (char *)malloc(strlen(name) + 15);
	if (space == (char *)NULL)
	{
		fprintf(stderr,"Failed to malloc in loading module %s\n",name);
		return((struct module_handle *)NULL);
	}
	it = (struct module_handle *)malloc(sizeof(struct module_handle));
	if (it == (struct module_handle *)NULL)
	{
		free(space);
		fprintf(stderr,"Failed to malloc in loading module %s\n",name);
		return((struct module_handle *)NULL);
	}
	it->name = strdup(name);
	if (it->name == (char *)NULL)
	{
		free(space);
		free(it);
		fprintf(stderr,"Failed to strdup in loading module %s\n",name);
		return((struct module_handle *)NULL);
	}
#ifndef WIN32
	sprintf(space, "./%s.so", name);
	dl_handle = dlopen(space, RTLD_LAZY);
#else
	sprintf(space, "%s.dll.", name);
	dl_handle = LoadLibrary(space);
#endif
#ifndef WIN32
	if (!dl_handle) {
		fprintf(stderr,"Failed to load module %s: %s\n",
			name, dlerror());
#else
	
	if (dl_handle==NULL) {
		fprintf(stderr,"Failed to load module %s: %i\n",
			name,GetLastError());
#endif
		free(it->name);
		free(it);
		free(space);
		return((struct module_handle *)NULL);
	}
	it->dl_handle = dl_handle;
	it->name = name;
	sprintf(space, "%s_supplies", name);
	it->supplies = extract_symbol(it, space);
	sprintf(space, "%s_uses", name);
	it->uses = extract_symbol(it, space);
	sprintf(space, "%s_use_functions", name);
	it->use_functions = extract_symbol(it, space);
	sprintf(space, "%s_initialize", name);
	it->initialize = extract_symbol(it, space);
	sprintf(space, "%s_start", name);
	it->start = extract_symbol(it, space);
	sprintf(space, "%s_stop", name);
	it->stop = extract_symbol(it, space);
	if ((it->supplies == NULL) ||
	    (it->uses == NULL) ||
	    (it->use_functions == NULL) ||
	    (it->initialize == NULL) ||
	    (it->start == NULL) ||
	    (it->stop == NULL))
	{
		fprintf(stderr,"Failed to load symbols for module %s.\n",
			name);
#ifndef WIN32
		dlclose(dl_handle);
#else
		FreeLibrary(dl_handle);
#endif
		free(it->name);
		free(it);
		free(space);
		return((struct module_handle *)NULL);
	}
	free(space);
	return(it);
}

/* a NULL return indicates failure to find/load */
struct module_handle *load_module(char *name)
{
	int i;

	for (i = 0; static_modules[i] != NULL; i++)
	{
		if (!strcmp(static_modules[i]->name,name))
		{
			return(static_modules[i]);
		}
	}
	return(dl_load_module(name));
}

/* TODO: fix to not dlclose static modules */
void unload_module(struct module_handle *it)
{
#ifndef WIN32
		dlclose(it);
#else
		FreeLibrary(it);
#endif
	free(it->name);
	free(it);
	return;
}

int main(int argc, char **argv)
{
	struct module_handle *config_handle;

	main_argc = argc;
	main_argv = argv;

	config_handle = load_module("config");
	if (config_handle == (struct module_handle *)NULL)
	{
		fprintf(stderr,"Couldn't load config module!  Aborting.\n");
		exit(-1);
	}
	if (config_handle->initialize() != 0)
	{
		fprintf(stderr,"Couldn't initialize config module!  Aborting.\n");
		exit(-1);
	}
	config_handle->start();
	{
		fprintf(stderr,"Couldn't start config module!  Aborting.\n");
		exit(-1);
	}
	unload_module(config_handle);
	exit(0);
#ifdef WIN32
	return 0;
#endif
}

/* TODO update to track loaded modules, and unload them on exit */
