/* Example DevMUD module to exercise interface standard and socket module */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include "../include/devmud.h"

/* TODO static number of players is very bad to assume */
#define MAX_PLAYERS 200 /* only 256 descriptors... */
#define READ_BUFFER_SIZE 16384
#define OUT_BUFFER_SIZE (READ_BUFFER_SIZE+256)
#define LASTLOG_SIZE 10

static void (*close_connection)(int playerid);
static void (*parsed_put_string)(int to, char *message);
static int (*get_message)(char *buffer, int size, int sec, int usec);
static int (*real_connection)(int playerid);
static char *(*connection_name)(int playerid);
static int (*connection_time)(int playerid);

static void *(*memory_allocate)(const char *module, unsigned int size);
static void (*memory_free)(const char *module, void *location);
static void (*memory_statistics)(const char *module, unsigned int *blocks, unsigned int *bytes);

char *str_dup(const char *s) {
  char *buf = memory_allocate("bigworld", strlen(s) + 1);
  strcpy(buf, s);
  return buf;
}


struct exit {
	char *name;
	struct object *dest;
	struct exit *next;
};

struct object {
	char *name;
	int player;
	struct object *where;
	struct object *inv;
	struct object *nextinv;
	struct exit *exit;
};

struct playerinfo {
	char name[100];
	char command_buffer[READ_BUFFER_SIZE];
	int command_tail;
	int state;
	int wrap;
	struct object *object;
} p[MAX_PLAYERS];

static struct object *root_object, *start_room;

static void send_string(int playerid, char *str)
{
	int wcount;
	char outbuf[OUT_BUFFER_SIZE*3];
	char *from, *to;

	if (p[playerid].wrap > 0)
	{
		to = outbuf;
		from = str;
		wcount = 0;
		while (*from)
		{
			if ((*to++ = *from++) == '\n')
			{
				wcount = 0;
			} else {
				wcount++;
			}
			if (wcount == p[playerid].wrap)
			{
				*to++ = '\r';
				*to++ = '\n';
				wcount = 0;
			}
		}
		*to = 0;
		parsed_put_string(playerid,outbuf);
	} else {
		parsed_put_string(playerid,str);
	}
}

static void say(char *str)
{
	int loop;

	for (loop=0;loop<MAX_PLAYERS;loop++)
	{
		if (real_connection(loop))
		{
			send_string(loop,str);
		}
	}
	/* For logging of the local discussion */
	fprintf(stderr,"%s",str);
}

static void tell_room(struct object *ob, char *str, struct object *ex1, struct object *ex2)
{
	struct object *t_ob;

	t_ob = ob->inv;
	while (t_ob != (struct object *)NULL)
	{
		if ((t_ob != ex1) &&
		    (t_ob != ex2))
		{
			if (t_ob->player != -1)
			{
				send_string(t_ob->player, str);
			}
		}
		t_ob = t_ob->nextinv;
	}
}

static void tell_other_rooms(struct object *root, struct object *ex, char *str)
{
	struct object *t_ob;

	t_ob = root->inv;
	while (t_ob != (struct object *)NULL)
	{
		if (t_ob != ex)
		{
			if (t_ob->player != -1)
			{
				send_string(t_ob->player, str);
			}
			tell_other_rooms(t_ob, ex, str);
		}
		t_ob = t_ob->nextinv;
	}
}

static struct object *new_object(void)
{
	struct object *this;

	this = (struct object *)memory_allocate("bigworld", sizeof(struct object));
	if (this == NULL)
	{
		fprintf(stderr,"Fatal: Out of memory.\n");
		exit(-1);
	}

	this->name = (char *)NULL;		/* not yet named */
	this->player = -1;			/* no connected player */
	this->where = (struct object *)NULL;	/* no location */
	this->inv = (struct object *)NULL;	/* no contents */
	this->nextinv = (struct object *)NULL;	/* nothing else in same place */
	this->exit = (struct exit *)NULL;	/* no exits yet */

	return(this);
}

/* don't call if it isn't in a location... */
static void remove_object(struct object *it)
{
	struct object *ob;

	if (it->where->inv == it)
	{
		it->where->inv = it->nextinv;
	}
	else
	{
		ob = it->where->inv;
		while (ob->nextinv != it)
		{
			ob = ob->nextinv;
		}
		ob->nextinv = it->nextinv;
	}
	it->where = (struct object *)NULL;
	it->nextinv = (struct object *)NULL;
}

/* requires that no exits currently point to it */
static void free_object(struct object *it)
{
	struct exit *ex, *t_ex;

	if (it->where != (struct object *)NULL)
	{
		remove_object(it);
	}
	/* remove inventory */
	while (it->inv != (struct object *)NULL)
	{
		free_object(it->inv);
	}
	if (it->name != (char *)NULL)
	{
		memory_free("bigworld", it->name);
	}
	ex = it->exit;
	while (ex != (struct exit *)NULL)
	{
		t_ex = ex->next;
		memory_free("bigworld", ex->name);
		memory_free("bigworld", ex);
		ex = t_ex;
	}
	memory_free("bigworld", it);
}

/* check for loop? */
static void insert_object(struct object *it, struct object *where)
{
	if (it->where != (struct object *)NULL)
	{
		remove_object(it);
	}
	it->where = where;
	it->nextinv = where->inv;
	where->inv = it;
}

static void link_exit(char *dir, struct object *here, struct object *there)
{
	struct exit *temp;

	temp = (struct exit *)memory_allocate("bigworld", sizeof(struct exit));
	if (temp == (struct exit *)NULL)
	{
		fprintf(stderr,"Fatal: Out of memory.\n");
		exit(-1);
	}
	temp->name = str_dup(dir);
	if (temp->name == (char *)NULL)
	{
		memory_free("bigword", temp);
		fprintf(stderr,"Fatal: Out of memory.\n");
		exit(-1);
	}
	temp->next = here->exit;
	temp->dest = there;
	here->exit = temp;
}

static void new_player(int playerid)
{
	char output_buffer[OUT_BUFFER_SIZE];
	struct object *playerob;

	playerob = new_object();

	p[playerid].name[0] = 0;
	p[playerid].state = 0;
	p[playerid].wrap = 0;
	p[playerid].command_tail = 0;
	p[playerid].object = playerob;
	playerob->player = playerid;
	insert_object(playerob, start_room);

	sprintf(output_buffer,"Player %d", playerid);
	sprintf(output_buffer,"Welcome, %d! (%s)\r\n", playerid, connection_name(playerid));
	say(output_buffer);
}

static void do_testcommand(int playerid, char *arg);
static void do_startchat(int playerid, char *arg);
static void do_startcommandchat(int playerid, char *arg);
static void do_endchat(int playerid, char *arg);
static void do_say(int playerid, char *arg);
static void do_emote(int playerid, char *arg);
static void do_nospace_emote(int playerid, char *arg);
static void do_help(int playerid, char *arg);
static void do_name(int playerid, char *arg);
static void do_who(int playerid, char *arg);
static void do_wrap(int playerid, char *arg);
static void do_quit(int playerid, char *arg);
static void do_where(int playerid, char *arg);
static void do_look(int playerid, char *arg);
static void do_go(int playerid, char *arg);
static void do_up(int playerid, char *arg);
static void do_down(int playerid, char *arg);
static void do_west(int playerid, char *arg);
static void do_east(int playerid, char *arg);
static void do_north(int playerid, char *arg);
static void do_south(int playerid, char *arg);
static void do_southeast(int playerid, char *arg);
static void do_southwest(int playerid, char *arg);
static void do_northeast(int playerid, char *arg);
static void do_northwest(int playerid, char *arg);
static void do_stats(int playerid, char *arg);

struct commandinfo {
	char *name;
	int length;
	int flags;
	void (*func)(int playerid, char *arg);
	char *helpstring;
} commands[] = {
	{ "chat", 4, 0, do_startchat,
		"\"chat\": say strings" },
	{ "commandchat", 11, 0, do_startcommandchat,
		"\"commandchat\": say strings that aren't commands" },
	{ ".", 1, 0, do_endchat,
		"\".\": end a chat mode" },
	{ "say ", 4, 1, do_say,
		"\"say \": say a string" },
	{ "\"", 1, 1, do_say,
		"\"\"\": say a string" },
	{ "'", 1, 1, do_say,
		"\"'\": say a string" },
	{ "emote ", 6, 1, do_emote,
		"\"emote \": emote a string" },
	{ ":", 1, 1, do_emote,
		"\":\": emote a string" },
	{ ";", 1, 1, do_nospace_emote,
		"\";\": emote a string with no extra space" },
	{ "tell ",5, 2, do_testcommand,
		/* TODO: something else */
		"\"tell \": tell another player a string (not working yet)" },
	{ "help", 4, 0, do_help,
		"\"help\": print commands" },
	{ "stats", 5, 0, do_stats,
		"\"stats\": print usage statistics" },
	{ "name ",5, 1, do_name,
		"\"name \": set your name" },
	{ "who", 3, 0, do_who,
		"\"who\": list connected players" },
	{ "w", 1, 0, do_who,
		"\"w\": alias for who" },
	{ "wrap ", 5, 1, do_wrap,
		"\"wrap \": set your connection to wrap at n characters" },
	{ "wrap", 4, 0, do_wrap,
		"\"wrap\" (with no argument): set your connection to not wrap" },
	{ "look", 4, 0, do_look,
		"\"look\": examine local environment" },
	{ "l", 1, 0, do_look,
		"\"l\": alias for look" },
	{ "quit", 4, 0, do_quit,
		"\"quit\": leave the game" },
	{ "where", 5, 0, do_where,
		"\"where\": tell what room you're in" },
	{ "go ", 3, 1, do_go,
		"\"go \": travel in some direction" },
	{ "up", 5, 0, do_up,
		"\"up\": travel up" },
	{ "down", 5, 0, do_down,
		"\"down\": travel down" },
	{ "north", 5, 0, do_north,
		"\"north\": travel north" },
	{ "south", 5, 0, do_south,
		"\"south\": travel south" },
	{ "east", 5, 0, do_east,
		"\"east\": travel east" },
	{ "west", 5, 0, do_west,
		"\"west\": travel west" },
	{ "northeast", 5, 0, do_northeast,
		"\"northeast\": travel northeast" },
	{ "northwest", 5, 0, do_northwest,
		"\"northwest\": travel northwest" },
	{ "southeast", 5, 0, do_southeast,
		"\"southeast\": travel southeast" },
	{ "southwest", 5, 0, do_southwest,
		"\"southwest\": travel southwest" },
	{ "test ", 5, 1, do_testcommand,
		"\"test\": a test command" },
	{ NULL, 0, -1, NULL,
		"\"\": no command" }
};

static void who(int playerid)
{
	char output_buffer[OUT_BUFFER_SIZE];
	int loop;
	int curtime, idletime, delta;
	char *timename;

	curtime = time(0L);
	send_string(playerid,"Player -1 (logger)\r\n");
	for (loop=0;loop<MAX_PLAYERS;loop++)
	{
		if (real_connection(loop))
		{
			timename = "second";
			idletime = connection_time(loop);
			delta = curtime - idletime;
			if (delta >= 60)
			{
				delta = delta/60;
				timename = "minute";
				if (delta >= 60)
				{
					delta = delta/60;
					timename = "hour";
					if (delta >= 24)
					{
						delta = delta/24;
						timename = "day";
						if (delta >= 7)
						{
							delta = delta/7;
							timename = "week";
						}
					}
				}
			}
			if (delta != 0)
			{
				sprintf(output_buffer,
					"Player %d%s, idle %d %s%s\r\n",
					loop, p[loop].name, delta, timename,
					((delta == 1) ? "" : "s"));
			} else {
				sprintf(output_buffer,"Player %d%s\r\n",
					loop, p[loop].name);
			}
			send_string(playerid,output_buffer);
		}
	}
}

static void do_testcommand(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer, "Test command: \"%s\"\r\n",arg);
	send_string(playerid, output_buffer);
}

static void do_startchat(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	p[playerid].state = 1;	/* player is in chat mode */
	sprintf(output_buffer, "Now in chat mode.  End with \".\" alone on a line.\r\n");
	send_string(playerid, output_buffer);
}

static void do_startcommandchat(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	if (p[playerid].state == 2)
	{
		sprintf(output_buffer, "Still in commandchat mode.  End with \".\" alone on a line.\r\n");
	} else {
		p[playerid].state = 2;	/* player is in commandchat mode */
		sprintf(output_buffer, "Now in commandchat mode.  End with \".\" alone on a line.\r\n");
	}
	send_string(playerid, output_buffer);
}

static void do_say(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"\r\nPlayer %d%s says: %s\r\n",
		playerid, p[playerid].name, arg);
	say(output_buffer);
}

static void do_emote(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"\r\nPlayer %d%s %s\r\n",
		playerid, p[playerid].name, arg);
	say(output_buffer);
}

static void do_nospace_emote(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"\r\nPlayer %d%s%s\r\n",
		playerid, p[playerid].name, arg);
	say(output_buffer);
}

static void do_who(int playerid, char *arg)
{
	who(playerid);
}

static void do_help(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	int index;

	send_string(playerid, "Commands are:\r\n");
	index = 0;
	while (commands[index].flags != -1)
	{
		sprintf(output_buffer,"%s.\r\n", commands[index].helpstring);
		send_string(playerid, output_buffer);
		index++;
	}
	send_string(playerid, "Big world version number is 2.\r\n");
}

static void do_stats(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	int blocks, bytes;

	send_string(playerid, "Memory usage:\r\n");

	memory_statistics("memorymanager", &blocks, &bytes);
	sprintf(output_buffer, "%-30s %10d %10d\r\n", "memorymanager", blocks, bytes);
	send_string(playerid, output_buffer);

	memory_statistics("socket", &blocks, &bytes);
	sprintf(output_buffer, "%-30s %10d %10d\r\n", "socket", blocks, bytes);
	send_string(playerid, output_buffer);

	memory_statistics("bigworld", &blocks, &bytes);
	sprintf(output_buffer, "%-30s %10d %10d\r\n", "bigworld", blocks, bytes);
	send_string(playerid, output_buffer);
}

static void do_wrap(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	int wraptemp;

	wraptemp = atoi(arg);
	if (wraptemp > 0)
	{
		sprintf(output_buffer,
			"Wrapping at %d characters.\r\n", wraptemp);
		p[playerid].wrap = wraptemp;
	} else {
		sprintf(output_buffer,"Wrapping disabled.\r\n");
		p[playerid].wrap = 0;
	}
	send_string(playerid, output_buffer);
}

static void do_name(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	if (strlen(arg) < 85)
	{
		sprintf(p[playerid].name," (%s)", arg);
		sprintf(output_buffer, "Now Player %d%s\r\n", playerid, p[playerid].name);
		send_string(playerid, output_buffer);
	}
}

static void do_endchat(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	if (p[playerid].state == 0)
	{
		sprintf(output_buffer, "Ending nonexistent chat mode.\r\n");
	} else {
		sprintf(output_buffer, "Ending chat mode.\r\n");
	}
	send_string(playerid, output_buffer);
	p[playerid].state = 0;
}

static void do_quit(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"\r\nPlayer %d%s quit.\r\n",playerid, p[playerid].name);
	say(output_buffer);
	free_object(p[playerid].object);
	close_connection(playerid);
}

static void do_where(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	struct object *ob;

	ob = p[playerid].object;
	ob = ob->where;
	sprintf(output_buffer,"In %s.\r\n", ob->name);
	send_string(playerid, output_buffer);
}


static void do_look(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	struct object *ob;
	struct exit *ex;

	ob = p[playerid].object;
	ob = ob->where;
	sprintf(output_buffer,"In %s.\r\n", ob->name);
	send_string(playerid, output_buffer);
	if (ob->exit)
	{
		sprintf(output_buffer,"Exits are:");
		ex = ob->exit;
		while (ex != (struct exit *)NULL)
		{
			strcat(output_buffer," ");
			strcat(output_buffer,ex->name);
			ex = ex->next;
		}
		strcat(output_buffer,"\r\n");
		send_string(playerid, output_buffer);
	}
}

static void do_go(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	struct object *ob, *room;
	struct exit *ex;

	ob = p[playerid].object;
	room = ob->where;
	ex = room->exit;
	while (ex != (struct exit *)NULL)
	{
		if (!strcmp(arg, ex->name))
		{
			sprintf(output_buffer,
				"\r\nPlayer %d%s leaves %s.\r\n",
				playerid, p[playerid].name, arg);
			tell_room(ob->where, output_buffer, ob, (struct object *)NULL);
			insert_object(ob, ex->dest);
			sprintf(output_buffer,
				"\r\nPlayer %d%s arrives.\r\n",
				playerid, p[playerid].name);
			tell_room(ex->dest, output_buffer, ob, (struct object *)NULL);
			sprintf(output_buffer, "You go %s.\r\n", arg);
			send_string(playerid, output_buffer);
			do_look(playerid, "");
			return;
		}
		ex = ex->next;
	}
	sprintf(output_buffer,"Can't go %s.\r\n", arg);
	send_string(playerid, output_buffer);
}

static void do_up(int playerid, char *arg)
{
	do_go(playerid,"up");
}

static void do_down(int playerid, char *arg)
{
	do_go(playerid,"down");
}

static void do_north(int playerid, char *arg)
{
	do_go(playerid,"north");
}

static void do_south(int playerid, char *arg)
{
	do_go(playerid,"south");
}

static void do_east(int playerid, char *arg)
{
	do_go(playerid,"east");
}

static void do_west(int playerid, char *arg)
{
	do_go(playerid,"west");
}

static void do_northeast(int playerid, char *arg)
{
	do_go(playerid,"northeast");
}

static void do_northwest(int playerid, char *arg)
{
	do_go(playerid,"northwest");
}

static void do_southeast(int playerid, char *arg)
{
	do_go(playerid,"southeast");
}

static void do_southwest(int playerid, char *arg)
{
	do_go(playerid,"southwest");
}

/* flags:
 * 1 = takes 1 arg...
 * -1 = end of list */

static int check_commandlist(int playerid, char *command)
{
	int index;

	index = 0;
	while (commands[index].flags != -1)
	{
		if (commands[index].flags & 0x1)	/* takes arg */
		{
			if (strncmp(command,
				    commands[index].name,
				    commands[index].length) == 0)
			{
				commands[index].func(playerid,
					command+commands[index].length);
				return(1);
			}
		} else {
			if (strcmp(command,commands[index].name) == 0)
			{
				commands[index].func(playerid,"");
				return(1);
			}
		}
		index++;
	}
	return(0);
}

static void parse_input(int playerid, char *buffer)
{
	char *command;
	char output_buffer[OUT_BUFFER_SIZE];

	/* read until newline, or... */
	while ((*buffer != '\n') && (*buffer != '\r') &&
	/* we really filled up the buffer. */
	       (p[playerid].command_tail < READ_BUFFER_SIZE))
	{
		p[playerid].command_buffer[p[playerid].command_tail] = *buffer;
		if (*buffer == 0)
		{
			return;
		}
		p[playerid].command_tail++;
		buffer++;
	}
	/* terminate the string, if needed */
	p[playerid].command_buffer[p[playerid].command_tail] = 0;
	/* and inform the next time that there isn't a partial buffer */
	p[playerid].command_tail = 0;
	command = p[playerid].command_buffer;

	if (p[playerid].state == 1)	/* chat mode */
	{
		if (strcmp(command,".") == 0)
		{
			p[playerid].state = 0;
			sprintf(output_buffer, "Ending chat mode.\r\n");
			send_string(playerid, output_buffer);
		} else {
			sprintf(output_buffer,"\r\nPlayer %d%s says: %s\r\n", playerid, p[playerid].name, command);
			say(output_buffer);
		}
	}
	else if (!check_commandlist(playerid,command))
	{
		/* unrecognized command case */
		if (*command)
		{
			if (p[playerid].state == 0)
			{
				sprintf(output_buffer,"Command \"%s\" unrecognized: try \"help\".\r\n",command);
				send_string(playerid, output_buffer);
			} else {
				sprintf(output_buffer,"\r\nPlayer %d%s says: %s\r\n", playerid, p[playerid].name, command);
				say(output_buffer);
			}
		}
	}
}

static void end_player(int playerid)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"Bye, %d!\r\n",playerid);
	say(output_buffer);
}

struct interface bigworld_supplies[] = {
	{NULL,NULL,NULL,NULL}
};

struct interface bigworld_uses[] = {
	{ "close_connection", NULL, "void(int)", NULL },
	{ "parsed_put_string", NULL, "void(int,char*)", NULL },
	{ "get_message", NULL, "int(char*,int,int,int)", NULL },
	{ "real_connection", NULL, "int(int)", NULL },
	{ "connection_name", NULL, "char*(int)", NULL },
	{ "connection_time", NULL, "int(int)", NULL },
	{ "config_port", NULL, "void(int)", NULL },
	{ "memory_allocate", NULL, "void*(const char*,unsigned int)", NULL },
	{ "memory_free", NULL, "void(const char*,void *)", NULL },
	{ "memory_statistics", NULL, "void(const char*,unsigned int*,unsigned int*)", NULL },
	{ NULL, NULL, NULL, NULL }
};

void bigworld_use_functions(struct interface *it)
{
	struct interface *temp;

	temp = it;
	while (temp->name != NULL)
	{
		if (!strcmp(temp->name,"close_connection"))
		{
			close_connection = temp->function;
		}
		else if (!strcmp(temp->name,"parsed_put_string"))
		{
			parsed_put_string = temp->function;
		}
		else if (!strcmp(temp->name,"get_message"))
		{
			get_message = temp->function;
		}
		else if (!strcmp(temp->name,"real_connection"))
		{
			real_connection = temp->function;
		}
		else if (!strcmp(temp->name,"connection_name"))
		{
			connection_name = temp->function;
		}
		else if (!strcmp(temp->name,"connection_time"))
		{
			connection_time = temp->function;
		}
		else if (!strcmp(temp->name,"memory_allocate"))
		{
			memory_allocate = temp->function;
		}
		else if (!strcmp(temp->name,"memory_free"))
		{
			memory_free = temp->function;
		}
		else if (!strcmp(temp->name,"memory_statistics"))
		{
			memory_statistics = temp->function;
		}
		temp++;
	}
	return;
}

int bigworld_initialize(void)
{
	return(0);
}

int bigworld_start(void)
{
	char read_buffer[READ_BUFFER_SIZE];
	int i, playerid;
	struct object *other_room;

	root_object = new_object();
	root_object->name = str_dup("root_object");
	start_room = new_object();
	start_room->name = str_dup("the start room");
	insert_object(start_room, root_object);
	other_room = new_object();
	other_room->name = str_dup("the other room");
	insert_object(other_room, root_object);

	link_exit("west", start_room, other_room);
	link_exit("east", other_room, start_room);

	/* initialize player data to known values */
	for(i=0;i<MAX_PLAYERS;i++)
	{
		p[i].command_buffer[0] = 0;
		p[i].command_tail = 0;
		p[i].state = 0;
		p[i].wrap = 0;
	}
	while(1)
	{
		read_buffer[0] = 0;
		while ((playerid = get_message(read_buffer,READ_BUFFER_SIZE,1,0)) == -1)
			;
		if (*read_buffer)
		{
			parse_input(playerid, read_buffer);
		} else {
			if (real_connection(playerid))
				new_player(playerid);
			else
				end_player(playerid);
		}
	}
	return(0);
}

int bigworld_stop(void)
{
	return(-1);
}
