/* Example DevMUD memory manager module. 
 * Originally written 4/10/99, by Mark Gritter <mgritter@cs.stanford.edu>
 *
 * This follows v0.2 of the DevMUD spec.
 *
 * Currently, this module only provides allocation and deallocation (via
 * malloc/free) and simple statistics about memory usage.
 *
 * There are lots of things that could be added here, like per-module 
 * memory limits, better statistics, reference counting/garbage collection
 * support, etc.
 */

#include "../include/devmud.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

struct mem_block {
  struct mem_block *next;
  void *where;
  unsigned int size;
};

struct usage_record {
  /* Usage records are stored in a linked list.  TODO: use a hash table
   * instead.
   */
  struct usage_record *next;

  const char *name;    /* Key for usage record is module name. */
  unsigned int bytes;  /* Number of bytes of memory allocated on heap. */
  unsigned int num;    /* Number of blocks. */

  /* This stores all the allocated blocks.  We're duplicating some of the
   * work malloc() does, but it lets us keep our records straight.
   * Note that memorymanager itself doesn't store individual allocations
   * here, to avoid recursion.
   * 
   * TODO: Turn this into a hashtable!
   */
  struct mem_block *blocks;
};

/* The head of the list will always be the record for 
 * "memorymanager" itself. 
 */
static struct usage_record *usage_records_head;

/* These functions are used when the memory manager itself places objects
 * on the heap. */
static void * self_allocate(unsigned int size) {
  usage_records_head->bytes += size;
  usage_records_head->num++;
  return malloc(size);
}

static void self_deallocate(void *block, unsigned int size) {
  usage_records_head->bytes -= size;
  usage_records_head->num--;
  free (block);
}

static struct mem_block * allocate_mem_block(void) {
  return (struct mem_block *)self_allocate(sizeof(struct mem_block));
}

static void deallocate_mem_block(struct mem_block *mb) {
  return self_deallocate(mb, sizeof(struct mem_block));
}

static struct usage_record * allocate_usage(void) {
  return (struct usage_record *)self_allocate(sizeof(struct usage_record));
}

static void deallocate_usage(struct usage_record *ur) {
  return self_deallocate(ur, sizeof(struct usage_record));
}

/* Generic allocation/deallocation functions for other modules */
static void * do_allocation(struct usage_record *record, 
                            unsigned int size) {
  struct mem_block *mb = allocate_mem_block();
  mb->next = record->blocks;
  mb->size = size;
  mb->where = malloc(size);
  record->bytes += size;
  record->num++;
  record->blocks = mb;
  return mb->where;
}

static void do_deallocation(struct usage_record *record,
                            void *storage) {
  struct mem_block *mb = record->blocks, *last = mb;
  while (mb != NULL) {
    if (mb->where == storage) {
      /* Remove from the linked list */
      if (mb == record->blocks) {
	record->blocks = mb->next;
      } else {
	last->next = mb->next;
      }
      free(mb->where);
      record->bytes -= mb->size;
      record->num--;
      deallocate_mem_block(mb);
      return;
    }
    last = mb;
    mb = mb->next;
  }
  fprintf(stderr, "Warning: module %s freed memory it did not own (%x).\n",
	  record->name, (unsigned int)storage);
}

static struct usage_record * find_module_usage_record(const char *name) {
  struct usage_record *curr = usage_records_head, *last = curr;
  while (curr != NULL) {
    if (strcmp(name, curr->name) == 0) return curr;
    last = curr;
    curr = curr->next;
  }

  /* Didn't find it in list, gotta create a new record */
  curr = allocate_usage();
  curr->name = self_allocate(strlen(name) + 1);
  strcpy((char *)curr->name, name);
  curr->bytes = 0;
  curr->num = 0;
  curr->blocks = NULL;

  /* Insert it in list. */
  curr->next = NULL;
  last->next = curr;
  return curr;
}

/* Now the publicly visible functions. */
static void *memory_allocate(const char *module, unsigned int size) {
  struct usage_record *record = find_module_usage_record(module);
  return do_allocation(record, size);
}

static void memory_free(const char *module, void *location) {
  struct usage_record *record;
  if (location == NULL) return;
  record = find_module_usage_record(module);
  return do_deallocation(record, location);
}

static void memory_statistics(const char *module, unsigned int *blocks, unsigned int *bytes) {
  struct usage_record *record = find_module_usage_record(module);
  *blocks = record->num;
  *bytes = record->bytes;
}

struct interface memorymanager_supplies[] = {
  { "memory_allocate", memory_allocate, 
    "void*(const char*,unsigned int)", NULL },
  { "memory_free", memory_free, 
    "void(const char*,void *)", NULL },
  { "memory_statistics", memory_statistics, 
    "void(const char*,unsigned int*,unsigned int*)", NULL },
  { NULL, NULL, NULL, NULL }
};

struct interface memorymanager_uses[] = {
  { NULL, NULL, NULL, NULL }
};

int memorymanager_initialize(void) {
  return 0;
}

int memorymanager_start(void) {
  usage_records_head = malloc(sizeof(struct usage_record));
  usage_records_head->name = "memorymanager";
  usage_records_head->bytes = sizeof(struct usage_record);
  usage_records_head->num = 1;
  usage_records_head->blocks = NULL;
  return 0;
}

int memorymanager_stop(void) {
  return -1;
}

void memorymanager_use_functions(struct interface *funcs) {
}
