#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#define TELOPTS
#include <arpa/telnet.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include "../include/devmud.h"

static int setup_socket(void);
static void cleanup_socket(void);
static void close_connection(int which);
static void put_string(int to, char *message, int length);
static void parsed_put_string(int to, char *message);
/* TODO
static void set_echo(int to, int on);
*/
static int get_message(char *buffer, int size, int sec, int usec);
static int real_connection(int which);
static char *connection_name(int which);
static int connection_time(int which);
static void config_port(int port);

static void *(*memory_allocate)(const char *module, unsigned int size);
static void (*memory_free)(const char *module, void *location);

/* limited by number of file descriptors */
#define MAX_CONNECTIONS 200

#define BACKLOG		5	/* number of pending conections.  OS limit */
#define MAXHOSTNAME	32	/* longest hostname dealt with. */
#define MAX_PACKET	512	/* with header should fit in 576 minimum */
#define FULL_MESSAGE	"All connections are taken.\n"
#define SITE_NAME_SIZE	80

struct interactive
{
	int closing;
	int socket;
	int last_time;
	struct sockaddr_in addr;
	char site_name[SITE_NAME_SIZE];
};

static int port_number = 2121;
extern int errno;
static struct interactive *connections[MAX_CONNECTIONS];
static fd_set readfds;
static int nfds = 0;
static int pending_close;
static int descriptor = -1;
static int opened = 0;

static int setup_socket(void)
{
	int parameter, i;
	struct sockaddr_in sa;
	struct hostent *hp;
	char localhost[MAXHOSTNAME+1];

	if (descriptor != -1)
	{
		cleanup_socket();
	}

	/* get our host name */
	if (gethostname(localhost, MAXHOSTNAME) == -1)
	{
		perror("Couldn't get hostname");
		return(0);
	}

	/* get host information from the hostname */
	if ((hp = gethostbyname(localhost)) == NULL)
	{
		perror("Couldn't get local host info");
		return(0);
	}

	/* set up our local address */
	/* this involves setting all of the fields in a sockaddr_in: */
	/* sin_family, sin_port, sin_addr, and sin_zero */

	/* this gets the sin_zero (reserved space) */
	memset((char *)&sa, 0, sizeof(sa));	/* to be safe */

	/* set the family */
	sa.sin_family = hp->h_addrtype;

	/* set the port.  This needs to be in network byte-order */
	sa.sin_port = htons(port_number);

	/* and set the address */
	/* copying the host address is workable, but misses loopback */
	/* bcopy((char *)hp->h_addr, (char *)&sa.sin_addr, hp->h_length); */
	/* instead, say we don't care: any address is ok */
	sa.sin_addr.s_addr = INADDR_ANY;

	/* create the socket */
	/* get a internet-based, two-way stream, protocol IP=0 */
	descriptor = socket(PF_INET,SOCK_STREAM,0);
	if (descriptor == -1)
	{
		perror("Couldn't get socket");
		return(0);
	}

	/* set the socket options to allow multiple sessions */
	parameter = 1;	/* 1 = turn on */
	if (setsockopt(descriptor, SOL_SOCKET, SO_REUSEADDR,
	               (char *) &parameter, sizeof(parameter)) < 0)
	{
		perror("Couldn't set socket option");
		close(descriptor);
		descriptor = -1;
		return(0);
	}

	/* bind the socket to an address */
	if (bind(descriptor, (struct sockaddr *)&sa, sizeof(sa)) < 0)
	{
		perror("Couldn't bind socket");
		close(descriptor);
		descriptor = -1;
		return(0);
	}

	/* setup to accept connections */
	if (listen(descriptor, BACKLOG) == -1)
	{
		perror("Couldn't setup accepting connections");
		close(descriptor);
		descriptor = -1;
		return(0);
	}

	/* ignore broken pipe errors */
	signal(SIGPIPE, SIG_IGN);

	if (!opened)
	{
		/* initialize some variables */
		for (i=0; i<MAX_CONNECTIONS; i++)
		{
			/* no connection for this index */
			connections[i] = NULL;
		}
		/* and no connections are waiting to be closed */
		pending_close = 0;
		opened = 1;
	}
	return(1);
}

static void cleanup_socket(void)
{
	if (descriptor != -1)
	{
		/* close the socket */
		close(descriptor);
		descriptor = -1;
	}
}

static void close_connection(int which)
{
	if (connections[which] == NULL)
	{
		fprintf(stderr,"Attempt to close a non connection.\n");
		return;
	}
	if (connections[which]->closing)
	{
		fprintf(stderr,"Attempt to close a closing connection.\n");
		return;
	}
	connections[which]->closing = 1;
	pending_close = 1;
	if (shutdown(connections[which]->socket, 2) == -1)
	{
		perror("shutdown socket");
		return;
	}
	if (close(connections[which]->socket) == -1)
	{
		perror("close socket");
		return;
	}
}

/* note that sending IAC is impolite in a telnet context, and the next
 * level driver should prevent that.  A number of other characters are
 * impolite as well */
static void put_string(int to, char *message, int length)
{
	int n;

	if ((connections[to] == NULL) || (connections[to]->closing))
	{
		return;
	}
	while(length>0)
	{
		n = write(connections[to]->socket, message, length > MAX_PACKET ? MAX_PACKET : length);
		if (n == -1)
		{
			switch(errno)
			{
				/* TODO send to an error channel */
				case EWOULDBLOCK:
				case EMSGSIZE:
				case EINVAL:
				case ENETUNREACH:
				case EHOSTUNREACH:
				case EPIPE:
					perror("write: socket closed");
					close_connection(to);
					return;
					/* don't loop back to here */
				default:
					perror("write to socket");
					close_connection(to);
					break;
			}
		} else if (n == 0) {
			fprintf(stderr,"Write: wrote no data!\n");
		} else {
			length -= n;
			message += n;
		}
	}
}

static void parsed_put_string(int to, char *message)
{
	static char buffer[1024];
	char *current;

	current = buffer;

	while (*message)
	{
		if (*message == -1)
		{
			*current++ = -1;
		}
		*current++ = *message++;
		if (current-buffer > 1020)
		{
			*current = 0;
			put_string(to,buffer,current-buffer);
			current = buffer;
		}
	}

	if (current != buffer)
	{
		put_string(to,buffer,current-buffer);
	}
}

/* TODO
static void set_echo(int to, int on)
{
	if (on)
	{
		put_string(to,"\xff\xfc\x01",3);
	}
	else
	{
		put_string(to,"\xff\xfb\x01",3);
	}
}
*/

static int get_message(char *buffer, int size, int sec, int usec)
{
	static int next = 0, next_close = 0;
	int i, socket, result;
	struct sockaddr_in sa;
	int length;
	struct timeval timeout;
	char temp[80];

	/* initialize to no message recieved */
	*buffer = 0;

	/* clean up any pending closes */
	if (pending_close)
	{
		i = next_close;
		do
		{
			if (connections[i] && connections[i]->closing)
			{
				next_close = (i+1)%MAX_CONNECTIONS;
				memory_free("socket", connections[i]);
				connections[i] = NULL;
				/* special case: return empty message=done */
				return(i);
			}
			i = (i+1)%MAX_CONNECTIONS;
		} while (i != next_close);
		pending_close = 0;
	}

	nfds = 0;
	FD_ZERO(&readfds);
	if (descriptor != -1)
	{
		FD_SET(descriptor, &readfds);
		nfds = descriptor+1;
	}
	for (i=0; i<MAX_CONNECTIONS; i++)
	{
		if (connections[i])
		{
			FD_SET(connections[i]->socket, &readfds);
			if (connections[i]->socket >= nfds)
				nfds = connections[i]->socket+1;
		}
	}
	timeout.tv_sec = sec;
	timeout.tv_usec = usec;
	result = select(nfds, &readfds, NULL, NULL, &timeout);
	if (result == -1)
	{
		if (errno == EINTR)
		{
			return -1;
		}
		perror("select");
	}
	else if (result == 0)
	{
		/* read nothing */
		return(-1);
	}
	if ((descriptor != -1) && (FD_ISSET(descriptor,&readfds)))
	{
		/* get a new socket */
		length = sizeof(sa);
		socket = accept(descriptor,(struct sockaddr *)&sa, &length);
		if (socket != -1)
		{
			for (i=0; i<MAX_CONNECTIONS; i++)
			{
				if (!connections[i])
				{
					if ((connections[i] = (struct interactive *)memory_allocate("socket", sizeof(struct interactive))) == NULL)
					{
						fprintf(stderr,"malloc failed.\n");
						exit(2);
					}
					connections[i]->closing = 0;
					connections[i]->socket = socket;
					connections[i]->last_time = time(0L);
					length = sizeof(struct sockaddr_in);
					getpeername(socket, (struct sockaddr*)&connections[i]->addr, &length);
					connections[i]->site_name[0] = 0;
					strcpy(connections[i]->site_name,inet_ntoa(connections[i]->addr.sin_addr));
					sprintf(temp, " port %d",connections[i]->addr.sin_port);
					strcat(connections[i]->site_name,temp);

					/* special case: return empty message=new */
					return(i);
				}
			}
			/* didn't find an open slot */
			write(socket, FULL_MESSAGE, strlen(FULL_MESSAGE));
			close(socket);
		}
		else
		{
			switch(errno)
			{
				case EINTR:
					break;
					return(-1); /* return nothing */
				default:
					/* TODO send to an error channel */
					perror("accept");
			}
		}
	}
	i = next;
	do
	{
		if (connections[i] && FD_ISSET(connections[i]->socket,&readfds))
		{
			next = (i+1)%MAX_CONNECTIONS;
			if ((length = read(connections[i]->socket, buffer, size - 1)) == -1)
			{
				switch(errno)
				{
					case ENETUNREACH:
					case EHOSTUNREACH:
					case ETIMEDOUT:
					case ECONNRESET:
					case EWOULDBLOCK:
					case EMSGSIZE:
						perror("read: socket closed");
						close_connection(i);
						break;
					default:
						/* TODO send to error channel */
						perror("read from socket");
						close_connection(i);
				}
			}
			else if (length == 0)
			{
				if (connections[i]->closing)
				{
					fprintf(stderr,"Tried to read from closing socket.\n");
					exit(5);
				}
				close_connection(i);
			}
			else /* length > 0 */
			{
				buffer[length] = 0;
				next = (i+1)%MAX_CONNECTIONS;
				connections[i]->last_time = time(0L);
				return(i);
			}
		}
		i = (i+1)%MAX_CONNECTIONS;
	} while (i != next);

	/* failed */
	return(-1);
}

static int real_connection(int which)
{
	if ((which < 0) || (which >= MAX_CONNECTIONS))
		return(0);
	return(connections[which] != NULL);
}

static int connection_time(int which)
{
	if ((which < 0) || (which >= MAX_CONNECTIONS))
		return (0);
	if (connections[which] != NULL)
		return(connections[which]->last_time);
	else
		return (0);
}

static char *connection_name(int which)
{
	if ((which < 0) || (which >= MAX_CONNECTIONS))
		return ("-");
	if (connections[which] != NULL)
		return(connections[which]->site_name);
	else
		return ("-");
}

static void config_port(int port)
{
	port_number = port;
}

struct interface socket_supplies[] = {
	{ "close_connection", close_connection, "void(int)", NULL },
	{ "parsed_put_string", parsed_put_string, "void(int,char*)", NULL },
	{ "get_message", get_message, "int(char*,int,int,int)", NULL },
	{ "real_connection", real_connection, "int(int)", NULL },
	{ "connection_name", connection_name, "char*(int)", NULL },
	{ "connection_time", connection_time, "int(int)", NULL },
	{ "config_port", config_port, "void(int)", NULL },
	{ NULL, NULL, NULL, NULL }
};

struct interface socket_uses[] = {
	{ "memory_allocate", NULL, "void*(const char*,unsigned int)", NULL },
	{ "memory_free", NULL, "void(const char*,void *)", NULL },
	{ NULL, NULL, NULL, NULL }
};

void socket_use_functions(struct interface *it)
{
  struct interface *temp;
  
  temp = it;
  while (temp->name != NULL) {
    if (!strcmp(temp->name,"memory_allocate")) {
      memory_allocate = temp->function;
    } else if (!strcmp(temp->name,"memory_free")) {
      memory_free = temp->function;
    }
    temp++;
  }
  return;
}

int socket_initialize(void)
{
	return(0);
}

int socket_start(void)
{
	setup_socket();

	return(0);
}

int socket_stop(void)
{
	return(-1);
}
