/*
 * assoc.c: functions for referring to structures by strings (associative array)
 */

/* includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "assoc.h"

/* calls to cleanup_data removed: struct data doesn't need cleanup here */

/* take a string and return the hash */
/* does not depend on which hash table is involved */
static unsigned int hash_key(char *key)
{
	unsigned int temp;

	temp = 0;
	while(*key)
	{
		temp = temp*43 + *key++;
	}
	return(temp);
}

/* returns NULL on malloc failure */
/* does not depend on which hash table is involved */
static struct bucket *make_string_bucket(char *key)
{
	struct bucket *temp;

	/* get the memory */
	temp = (struct bucket *) malloc(sizeof(struct bucket));

	/* and check it */
	if (temp == (struct bucket *)NULL)
	{
		fprintf(stderr,"Couldn't allocate memory for a new bucket.\n");
		return((struct bucket *)NULL);
	}
	temp->hashvalue = hash_key(key);
	temp->keytype = STRING;
	temp->key.text = strdup(key);
	if (temp->key.text == (char *)NULL)
	{
		fprintf(stderr,"Couldn't allocate memory for a new key.\n");
		free(temp);
		return((struct bucket *)NULL);
	}
	return(temp);
}

/* returns NULL on malloc failure */
/* does not depend on which hash table is involved */
static struct bucket *make_number_bucket(int key)
{
	struct bucket *temp;

	/* get the memory */
	temp = (struct bucket *) malloc(sizeof(struct bucket));

	/* and check it */
	if (temp == (struct bucket *)NULL)
	{
		fprintf(stderr,"Couldn't allocate memory for a new bucket.\n");
		return((struct bucket *)NULL);
	}
	temp->hashvalue = key;
	temp->keytype = NUMBER;
	temp->key.val = key;
	return(temp);
}

int init_assoc(struct hash *h, int buckets)
{
	int loop;

	/* init the important variables */
	h->count = 0;
	h->collisions = 0;
	h->num_buckets = buckets-1;	/* num_buckets is low by 1: & trick */

	/* sanity check */
	if (h->num_buckets < 0)
	{
		fprintf(stderr,"Must have at least one bucket for hashing.\n");
		return(-1);
	}

	/* get the memory */
	h->all_buckets = (struct bucket **)
			malloc((h->num_buckets+1)*sizeof(struct bucket *));

	/* and check it */
	if (h->all_buckets == (struct bucket **)NULL)
	{
		fprintf(stderr,"Couldn't allocate memory for buckets.\n");
		return(-1);
	}

	/* initialize to nothing */
	for(loop=0; loop<=h->num_buckets; loop++)
	{
		h->all_buckets[loop] = (struct bucket *)NULL;
	}
	return(0);	/* success */
}

void cleanup_assoc(struct hash *h)
{
	int loop;
	struct bucket *temp;

	for(loop=0; loop<=h->num_buckets; loop++)
	{
		temp = h->all_buckets[loop];
		while (temp != (struct bucket *)NULL)
		{
			h->all_buckets[loop] = h->all_buckets[loop]->next;
			if (temp->keytype == STRING)
			{
				free(temp->key.text);
			}
			free(temp);
			temp = h->all_buckets[loop];
		}
	}
}

/* returns the appropriate bucket, creating it if it didn't exist */
struct bucket *new_string_bucket(struct hash *h, char *key)
{
	unsigned int hashvalue;
	struct bucket *temp;

	hashvalue = hash_key(key);
	temp = h->all_buckets[hashvalue&h->num_buckets];
	while (temp != (struct bucket *)NULL)
	{
		if ((temp->keytype == STRING) &&
		    (hashvalue == temp->hashvalue) &&
		    !strcmp(key,temp->key.text))
		{
			/* found it */
			return(temp);
		}
		temp = temp->next;
	}
	if (h->all_buckets[hashvalue&h->num_buckets] != (struct bucket *)NULL)
	{
		if (h->collisions > h->num_buckets)
			resize_assoc(h, h->num_buckets * 2 + 2);
		else
			h->collisions++;
	}

	/* allocate a new bucket... */
	temp = make_string_bucket(key);

	/* if we got one */
	if (temp != (struct bucket *)NULL)
	{
		h->count++;
		temp->next = h->all_buckets[hashvalue&h->num_buckets];
		h->all_buckets[hashvalue&h->num_buckets] = temp;
	}
	return(temp);	/* which is NULL if we failed */
}

/* returns the appropriate bucket, creating it if it didn't exist */
struct bucket *new_number_bucket(struct hash *h, int key)
{
	unsigned int hashvalue;
	struct bucket *temp;

	hashvalue = key;
	temp = h->all_buckets[hashvalue&h->num_buckets];
	while (temp != (struct bucket *)NULL)
	{
		if ((temp->keytype == NUMBER) &&
		    (hashvalue == temp->hashvalue) &&
		    (key == temp->key.val))
		{
			/* found it */
			return(temp);
		}
		temp = temp->next;
	}
	if (h->all_buckets[hashvalue&h->num_buckets] != (struct bucket *)NULL)
	{
		if (h->collisions > h->num_buckets)
			resize_assoc(h, h->num_buckets * 2 + 2);
		else
			h->collisions++;
	}

	/* allocate a new bucket... */
	temp = make_number_bucket(key);

	/* if we got one */
	if (temp != (struct bucket *)NULL)
	{
		h->count++;
		temp->next = h->all_buckets[hashvalue&h->num_buckets];
		h->all_buckets[hashvalue&h->num_buckets] = temp;
	}
	return(temp);	/* which is NULL if we failed */
}

/* return the bucket only if it does exist */
struct bucket *find_string_bucket(struct hash *h, char *key)
{
	unsigned int hashvalue;
	struct bucket *temp;

	hashvalue = hash_key(key);
	temp = h->all_buckets[hashvalue&h->num_buckets];
	while (temp != (struct bucket *)NULL)
	{
		if ((temp->keytype == STRING) &&
		    (hashvalue == temp->hashvalue) &&
		    !strcmp(key,temp->key.text))
		{
			return(temp);
		}
		temp = temp->next;
	}
	return((struct bucket *)NULL);
}

/* return the bucket only if it does exist */
struct bucket *find_number_bucket(struct hash *h, int key)
{
	unsigned int hashvalue;
	struct bucket *temp;

	hashvalue = key;
	temp = h->all_buckets[hashvalue&h->num_buckets];
	while (temp != (struct bucket *)NULL)
	{
		if ((temp->keytype == NUMBER) &&
		    (hashvalue == temp->hashvalue) &&
		    (key == temp->key.val))
		{
			return(temp);
		}
		temp = temp->next;
	}
	return((struct bucket *)NULL);
}

/* delete the bucket if it exists */
int delete_string_bucket(struct hash *h, char *key)
{
	unsigned int hashvalue;
	struct bucket *temp, **parent;

	hashvalue = hash_key(key);
	parent = &(h->all_buckets[hashvalue&h->num_buckets]);
	temp = *parent;
	while (temp != (struct bucket *)NULL)
	{
		if ((temp->keytype == STRING) &&
		    (hashvalue == temp->hashvalue) &&
		    !strcmp(key,temp->key.text))
		{
			/* found it, so remove it. */
			*parent = temp->next;
			if (temp->next != (struct bucket *)NULL)
			{
				h->collisions--;
			}
			h->count--;
			free(temp->key.text);
			free(temp);
			return(1);
		}
		parent = &(temp->next);
		temp = *parent;
	}
	return(0);
}

/* delete the bucket if it exists */
int delete_number_bucket(struct hash *h, int key)
{
	unsigned int hashvalue;
	struct bucket *temp, **parent;

	hashvalue = key;
	parent = &(h->all_buckets[hashvalue&h->num_buckets]);
	temp = *parent;
	while (temp != (struct bucket *)NULL)
	{
		if ((temp->keytype == NUMBER) &&
		    (hashvalue == temp->hashvalue) &&
		    (key == temp->key.val))
		{
			/* found it, so remove it. */
			*parent = temp->next;
			if (temp->next != (struct bucket *)NULL)
			{
				h->collisions--;
			}
			h->count--;
			free(temp);
			return(1);
		}
		parent = &(temp->next);
		temp = *parent;
	}
	return(0);
}

int resize_assoc(struct hash *h, int buckets)
{
	struct bucket *temp, **old_buckets;
	int old_num_buckets, old_collisions, loop;

	/* sanity check */
	if (buckets < 1)
	{
		fprintf(stderr,"Must have at least one bucket for hashing.\n");
		return(-1);
	}
	/* check for power of two */
	if (buckets & (buckets-1))
	{
		fprintf(stderr,"Number of buckets must be a power of two.\n");
		return(-1);
	}

	old_num_buckets = h->num_buckets;
	old_buckets = h->all_buckets;
	old_collisions = h->collisions;

	h->collisions = 0;
	h->num_buckets = buckets-1;

	/* get the memory */
	h->all_buckets = (struct bucket **)
			malloc((h->num_buckets+1)*sizeof(struct bucket *));

	/* and check it */
	if (h->all_buckets == (struct bucket **)NULL)
	{
		fprintf(stderr,"Couldn't allocate memory for more buckets.\n");
		/* put it back... */
		h->num_buckets = old_num_buckets;
		h->all_buckets = old_buckets;
		h->collisions = old_collisions;
		return(-1);
	}

	/* initialize to nothing -- hashes won't be the same */
	for(loop=0; loop<=h->num_buckets; loop++)
	{
		h->all_buckets[loop] = (struct bucket *)NULL;
	}

	/* now move stuff over */
	for (loop=0; loop<=old_num_buckets; loop++)
	{
		temp = old_buckets[loop];
		while (temp != (struct bucket *)NULL)
		{
			old_buckets[loop] = temp->next;
			
			if (h->all_buckets[temp->hashvalue&h->num_buckets] !=
					(struct bucket *)NULL)
				h->collisions++;
			temp->next = h->all_buckets[temp->hashvalue&h->num_buckets];
			h->all_buckets[temp->hashvalue&h->num_buckets] = temp;
			temp = old_buckets[loop];
		}
	}
	free(old_buckets);

	return(0);	/* success */
}
