/* Example DevMUD module to exercise interface standard and socket module */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include "../include/devmud.h"
#include "assoc.h"

#define SNEAKY_FLAG 1

struct hash idtable;

/* TODO make player struct dynamic, check for more players ok in select */
#define MAX_PLAYERS 200
#define MAX_MOBS 4
#define READ_BUFFER_SIZE 16384
#define OUT_BUFFER_SIZE (READ_BUFFER_SIZE+256)
#define LASTLOG_SIZE 10

static void (*queue_event)(struct timeval when, void (*func)(struct timeval when, int data), int data) = NULL;
static void (*close_connection)(int which) = NULL;
static int (*real_connection)(int which) = NULL;
static char *(*connection_name)(int which) = NULL;
static struct timeval (*connection_time)(int which) = NULL;
static int (*open_port)
		(int port,
		 unsigned int addr,
		 int (*new_connection)
		       (int id,
			void (*message)(int id,char *message,int len,int oob)),
		 void (*end_connection)(int id),
		 void (*message)(int id,char *message,int len,int oob))
	= NULL;
static void (*close_port)(int id) = NULL;

/* TODO cleanup state on player exit to mark as avail */

struct exit {
	char *name;
	struct object *dest;
	struct exit *next;
};

struct object {
	char *name;
	int player;
	struct object *where;
	struct object *inv;
	struct object *nextinv;
	struct exit *exit;
	int flags;	/* TODO: something more portable */
};

struct playerinfo {
	/* function to send messages */
	void (*message)(int id,char *message,int len,int oob);
	int id;	/* network ID */
	char command_buffer[READ_BUFFER_SIZE];
	int command_tail;
	int state;
	int wrap;
	struct object *object;
};

static struct playerinfo p[MAX_PLAYERS];

struct mobinfo {
	struct object *object;
	int delay;
};

static struct mobinfo m[MAX_MOBS];

static struct object *root_object, *start_room;

static int port_number = 2121;

static void send_string(int playerid, char *str)
{
	int wcount;
	char outbuf[OUT_BUFFER_SIZE*3];
	char *from, *to;

	if (p[playerid].wrap > 0)
	{
		to = outbuf;
		from = str;
		wcount = 0;
		while (*from)
		{
			if ((*to++ = *from++) == '\n')
			{
				wcount = 0;
			} else {
				wcount++;
			}
			if (wcount == p[playerid].wrap)
			{
				*to++ = '\r';
				*to++ = '\n';
				wcount = 0;
			}
		}
		*to = 0;
		p[playerid].message(p[playerid].id,outbuf,-1,0);
	} else {
		p[playerid].message(p[playerid].id,str,-1,0);
	}
}

static void mob_send_string(struct object *mob, char *str)
{
	if (mob->player != -1)
	{
		send_string(mob->player, str);
	}
}

/* TODO: don't use this... */
static void say(char *str)
{
	int loop;

	for (loop=0;loop<MAX_PLAYERS;loop++)
	{
		if (p[loop].state != -1)
		{
			send_string(loop,str);
		}
	}
	/* For logging of the local discussion */
	fprintf(stderr,"%s",str);
}

static void tell_room(struct object *ob, char *str, struct object *ex1, struct object *ex2)
{
	struct object *t_ob;

	t_ob = ob->inv;
	while (t_ob != (struct object *)NULL)
	{
		if ((t_ob != ex1) &&
		    (t_ob != ex2))
		{
			if (t_ob->player != -1)
			{
				send_string(t_ob->player, str);
			}
		}
		t_ob = t_ob->nextinv;
	}
}

static void tell_other_rooms(struct object *root, struct object *ex, char *str)
{
	struct object *t_ob;

	t_ob = root->inv;
	while (t_ob != (struct object *)NULL)
	{
		if (t_ob != ex)
		{
			if (t_ob->player != -1)
			{
				send_string(t_ob->player, str);
			}
			tell_other_rooms(t_ob, ex, str);
		}
		t_ob = t_ob->nextinv;
	}
}

static struct object *new_object(void)
{
	struct object *this;

	this = (struct object *)malloc(sizeof(struct object));
	if (this == NULL)
	{
		fprintf(stderr,"Fatal: Out of memory.\n");
		exit(-1);
	}

	this->name = (char *)NULL;		/* not yet named */
	this->player = -1;			/* no connected player */
	this->where = (struct object *)NULL;	/* no location */
	this->inv = (struct object *)NULL;	/* no contents */
	this->nextinv = (struct object *)NULL;	/* nothing else in same place */
	this->exit = (struct exit *)NULL;	/* no exits yet */
	this->flags = 0;			/* no flags */

	return(this);
}

/* don't call if it isn't in a location... */
static void remove_object(struct object *it)
{
	struct object *ob;

	if (it->where->inv == it)
	{
		it->where->inv = it->nextinv;
	}
	else
	{
		ob = it->where->inv;
		while (ob->nextinv != it)
		{
			ob = ob->nextinv;
		}
		ob->nextinv = it->nextinv;
	}
	it->where = (struct object *)NULL;
	it->nextinv = (struct object *)NULL;
}

/* requires that no exits currently point to it */
static void free_object(struct object *it)
{
	struct exit *ex, *t_ex;

	if (it->where != (struct object *)NULL)
	{
		remove_object(it);
	}
	/* remove inventory */
	while (it->inv != (struct object *)NULL)
	{
		free_object(it->inv);
	}
	if (it->name != (char *)NULL)
	{
		free(it->name);
	}
	ex = it->exit;
	while (ex != (struct exit *)NULL)
	{
		t_ex = ex->next;
		free(ex->name);
		free(ex);
		ex = t_ex;
	}
	free(it);
}

/* check for loop? */
static void insert_object(struct object *it, struct object *where)
{
	if (it->where != (struct object *)NULL)
	{
		remove_object(it);
	}
	it->where = where;
	it->nextinv = where->inv;
	where->inv = it;
}

static void link_exit(char *dir, struct object *here, struct object *there)
{
	struct exit *temp;

	temp = (struct exit *)malloc(sizeof(struct exit));
	if (temp == (struct exit *)NULL)
	{
		fprintf(stderr,"Fatal: Out of memory.\n");
		exit(-1);
	}
	temp->name = strdup(dir);
	if (temp->name == (char *)NULL)
	{
		free(temp);
		fprintf(stderr,"Fatal: Out of memory.\n");
		exit(-1);
	}
	temp->next = here->exit;
	temp->dest = there;
	here->exit = temp;
}

static int new_player(int id,void (*message)(int id,char *message,int len,int oob))
{
	char output_buffer[OUT_BUFFER_SIZE];
	struct object *playerob;
	int playerid;

	/* find a playerid */
	for (playerid=0;
	     (playerid<MAX_PLAYERS) && (p[playerid].state != -1);
	     playerid++)
		;	/* do nothing */
	if (playerid >= MAX_PLAYERS)
	{
		/* TODO bail on player */
	}
	
	playerob = new_object();

	p[playerid].state = 0;
	p[playerid].wrap = 0;
	p[playerid].command_tail = 0;
	p[playerid].object = playerob;
	p[playerid].message = message;
	p[playerid].id = id;
	playerob->player = playerid;
	sprintf(output_buffer,"Player %d", playerid);
	playerob->name = strdup(output_buffer);
	insert_object(playerob, start_room);

	sprintf(output_buffer,"Welcome, %d! (%s)\r\n", playerid, connection_name(p[playerid].id));
	say(output_buffer);
	return(playerid);	/* what ID to use to contact us */
}

static void do_spew(int playerid, char *arg);
static void do_testcommand(int playerid, char *arg);
static void do_startchat(int playerid, char *arg);
static void do_startcommandchat(int playerid, char *arg);
static void do_endchat(int playerid, char *arg);
static void do_say(int playerid, char *arg);
static void do_emote(int playerid, char *arg);
static void do_nospace_emote(int playerid, char *arg);
static void do_help(int playerid, char *arg);
static void do_name(int playerid, char *arg);
static void do_who(int playerid, char *arg);
static void do_wrap(int playerid, char *arg);
static void do_quit(int playerid, char *arg);
static void do_where(int playerid, char *arg);
static void do_look(int playerid, char *arg);
static void do_go(int playerid, char *arg);
static void do_up(int playerid, char *arg);
static void do_down(int playerid, char *arg);
static void do_west(int playerid, char *arg);
static void do_east(int playerid, char *arg);
static void do_north(int playerid, char *arg);
static void do_south(int playerid, char *arg);
static void do_southeast(int playerid, char *arg);
static void do_southwest(int playerid, char *arg);
static void do_northeast(int playerid, char *arg);
static void do_northwest(int playerid, char *arg);

struct commandinfo {
	char *name;
	int length;
	int flags;
	void (*func)(int playerid, char *arg);
	char *helpstring;
} commands[] = {
	{ "chat", 4, 0, do_startchat,
		"\"chat\": say strings" },
	{ "commandchat", 11, 0, do_startcommandchat,
		"\"commandchat\": say strings that aren't commands" },
	{ ".", 1, 0, do_endchat,
		"\".\": end a chat mode" },
	{ "say ", 4, 1, do_say,
		"\"say \": say a string" },
	{ "\"", 1, 1, do_say,
		"\"\"\": say a string" },
	{ "'", 1, 1, do_say,
		"\"'\": say a string" },
	{ "emote ", 6, 1, do_emote,
		"\"emote \": emote a string" },
	{ ":", 1, 1, do_emote,
		"\":\": emote a string" },
	{ ";", 1, 1, do_nospace_emote,
		"\";\": emote a string with no extra space" },
	{ "tell ",5, 2, do_testcommand,
		/* TODO: something else */
		"\"tell \": tell another player a string (not working yet)" },
	{ "help", 4, 0, do_help,
		"\"help\": print commands" },
	{ "name ",5, 1, do_name,
		"\"name \": set your name" },
	{ "who", 3, 0, do_who,
		"\"who\": list connected players" },
	{ "w", 1, 0, do_who,
		"\"w\": alias for who" },
	{ "wrap ", 5, 1, do_wrap,
		"\"wrap \": set your connection to wrap at n characters" },
	{ "wrap", 4, 0, do_wrap,
		"\"wrap\" (with no argument): set your connection to not wrap" },
	{ "look", 4, 0, do_look,
		"\"look\": examine local environment" },
	{ "l", 1, 0, do_look,
		"\"l\": alias for look" },
	{ "quit", 4, 0, do_quit,
		"\"quit\": leave the game" },
	{ "where", 5, 0, do_where,
		"\"where\": tell what room you're in" },
	{ "go ", 3, 1, do_go,
		"\"go \": travel in some direction" },
	{ "up", 5, 0, do_up,
		"\"up\": travel up" },
	{ "down", 5, 0, do_down,
		"\"down\": travel down" },
	{ "north", 5, 0, do_north,
		"\"north\": travel north" },
	{ "south", 5, 0, do_south,
		"\"south\": travel south" },
	{ "east", 5, 0, do_east,
		"\"east\": travel east" },
	{ "west", 5, 0, do_west,
		"\"west\": travel west" },
	{ "northeast", 5, 0, do_northeast,
		"\"northeast\": travel northeast" },
	{ "northwest", 5, 0, do_northwest,
		"\"northwest\": travel northwest" },
	{ "southeast", 5, 0, do_southeast,
		"\"southeast\": travel southeast" },
	{ "southwest", 5, 0, do_southwest,
		"\"southwest\": travel southwest" },
	{ "test ", 5, 1, do_testcommand,
		"\"test\": a test command" },
	{ "spew ", 5, 1, do_spew,
		"\"spew\": try to overflow output" },
	{ NULL, 0, -1, NULL,
		"\"\": no command" }
};

static void who(int playerid)
{
	char output_buffer[OUT_BUFFER_SIZE];
	int loop, secdelta, usecdelta;
	struct timeval idletime;
	char *timename;

	send_string(playerid,"Player -1 (logger)\r\n");
	for (loop=0;loop<MAX_PLAYERS;loop++)
	{
		if (p[loop].state != -1)
		{
			timename = "second";
			idletime = connection_time(p[loop].id);
			usecdelta = current_time.tv_usec - idletime.tv_usec;
			secdelta = current_time.tv_sec - idletime.tv_sec;
			if (usecdelta < 0)
			{
				secdelta--;
				usecdelta += 1000000;
			}
			if (secdelta >= 60)
			{
				secdelta = secdelta/60;
				timename = "minute";
				if (secdelta >= 60)
				{
					secdelta = secdelta/60;
					timename = "hour";
					if (secdelta >= 24)
					{
						secdelta = secdelta/24;
						timename = "day";
						if (secdelta >= 7)
						{
							secdelta = secdelta/7;
							timename = "week";
						}
					}
				}
			}
			if (secdelta != 0)
			{
				sprintf(output_buffer,
					"%s, idle %d %s%s\r\n",
					p[loop].object->name, secdelta, timename,
					((secdelta == 1) ? "" : "s"));
			} else if (usecdelta != 0) {
				sprintf(output_buffer,
					"%s, idle %d microsecond%s\r\n",
					p[loop].object->name, usecdelta,
					((usecdelta == 1) ? "" : "s"));
			} else {
				sprintf(output_buffer,"%s\r\n",
					p[loop].object->name);
			}
			send_string(playerid,output_buffer);
		}
	}
}

static void do_testcommand(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer, "Test command: \"%s\"\r\n",arg);
	send_string(playerid, output_buffer);
}

static void do_spew(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	int i;

	sprintf(output_buffer, "Spew line: \"%s\"\r\n",arg);
	for (i=0; i<4096; i++)
		send_string(playerid, output_buffer);
}

static void do_startchat(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	p[playerid].state = 1;	/* player is in chat mode */
	sprintf(output_buffer, "Now in chat mode.  End with \".\" alone on a line.\r\n");
	send_string(playerid, output_buffer);
}

static void do_startcommandchat(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	if (p[playerid].state == 2)
	{
		sprintf(output_buffer, "Still in commandchat mode.  End with \".\" alone on a line.\r\n");
	} else {
		p[playerid].state = 2;	/* player is in commandchat mode */
		sprintf(output_buffer, "Now in commandchat mode.  End with \".\" alone on a line.\r\n");
	}
	send_string(playerid, output_buffer);
}

static void do_say(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"\r\n%s says: %s\r\n",
		p[playerid].object->name, arg);
	say(output_buffer);
}

static void do_emote(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"\r\n%s %s\r\n",
		p[playerid].object->name, arg);
	say(output_buffer);
}

static void do_nospace_emote(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"\r\n%s%s\r\n",
		p[playerid].object->name, arg);
	say(output_buffer);
}

static void do_who(int playerid, char *arg)
{
	who(playerid);
}

static void do_help(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	int index;

	send_string(playerid, "Commands are:\r\n");
	index = 0;
	while (commands[index].flags != -1)
	{
		sprintf(output_buffer,"%s.\r\n", commands[index].helpstring);
		send_string(playerid, output_buffer);
		index++;
	}
	send_string(playerid, "Big world version number is 7.\r\n");
}

static void do_wrap(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	int wraptemp;

	wraptemp = atoi(arg);
	if (wraptemp > 0)
	{
		sprintf(output_buffer,
			"Wrapping at %d characters.\r\n", wraptemp);
		p[playerid].wrap = wraptemp;
	} else {
		sprintf(output_buffer,"Wrapping disabled.\r\n");
		p[playerid].wrap = 0;
	}
	send_string(playerid, output_buffer);
}

static void do_name(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	if (strlen(arg) < 85)
	{
		if (arg[0])
		{
			sprintf(output_buffer,"Player %d (%s)", playerid, arg);
		} else {
			sprintf(output_buffer,"Player %d", playerid);
		}
		free(p[playerid].object->name);
		p[playerid].object->name = strdup(output_buffer);
		sprintf(output_buffer, "Now %s\r\n", p[playerid].object->name);
		send_string(playerid, output_buffer);
	}
}

static void do_endchat(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	if (p[playerid].state == 0)
	{
		sprintf(output_buffer, "Ending nonexistent chat mode.\r\n");
	} else {
		sprintf(output_buffer, "Ending chat mode.\r\n");
	}
	send_string(playerid, output_buffer);
	p[playerid].state = 0;
}

static void do_quit(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];

	sprintf(output_buffer,"\r\n%s quit.\r\n",p[playerid].object->name);
	say(output_buffer);
	if (p[playerid].object != (struct object *)NULL)
		free_object(p[playerid].object);
	p[playerid].object = (struct object *)NULL;
	close_connection(p[playerid].id);
}

static void do_where(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	struct object *ob;

	ob = p[playerid].object;
	ob = ob->where;
	sprintf(output_buffer,"In %s.\r\n", ob->name);
	send_string(playerid, output_buffer);
}

static void do_look(int playerid, char *arg)
{
	char output_buffer[OUT_BUFFER_SIZE];
	struct object *ob, *room, *others;
	struct exit *ex;

	ob = p[playerid].object;
	room = ob->where;
	sprintf(output_buffer,"In %s.\r\n", room->name);
	send_string(playerid, output_buffer);
	if (room->exit)
	{
		sprintf(output_buffer,"Exits are:");
		ex = room->exit;
		while (ex != (struct exit *)NULL)
		{
			strcat(output_buffer," ");
			strcat(output_buffer,ex->name);
			ex = ex->next;
		}
		strcat(output_buffer,"\r\n");
		send_string(playerid, output_buffer);
		others = room->inv;
		while (others != (struct object *)NULL)
		{
			if (ob != others)
			{
				sprintf(output_buffer,"%s is here.\r\n", others->name);
				send_string(playerid, output_buffer);
			}
			others = others->nextinv;
		}
	}
}

static void mob_look(struct object *mob)
{
	if (mob->player != -1)
	{
		do_look(mob->player, "");
	}
}

static char *random_exit(struct object *room)
{
	int exitcount, exitchoice;
	struct exit *tempexit;

	exitcount = 0;
	tempexit = room->exit;
	while (tempexit != (struct exit *)NULL)
	{
		exitcount++;
		tempexit = tempexit->next;
	}
	
	if (exitcount == 0)
	{
		return("");
	}
	exitchoice = lrand48() % exitcount;
	tempexit = room->exit;
	while (exitchoice > 0)
	{
		exitchoice--;
		tempexit = tempexit->next;
	}
	return(tempexit->name);
}

static void mob_go(struct object *mob, char *dir)
{
	char output_buffer[OUT_BUFFER_SIZE];
	struct object *room;
	struct exit *ex;

	room = mob->where;
	ex = room->exit;
	while (ex != (struct exit *)NULL)
	{
		if (!strcmp(dir, ex->name))
		{
			if (!(mob->flags & SNEAKY_FLAG))
			{
				sprintf(output_buffer, "\r\n%s leaves %s.\r\n",
							mob->name, ex->name);
				tell_room(mob->where, output_buffer,
					mob, (struct object *)NULL);
			}
			insert_object(mob, ex->dest);
			if (!(mob->flags & SNEAKY_FLAG))
			{
				sprintf(output_buffer, "\r\n%s arrives.\r\n",
					mob->name);
				tell_room(ex->dest, output_buffer,
					mob, (struct object *)NULL);
			}
			sprintf(output_buffer, "You go %s.\r\n", dir);
			mob_send_string(mob, output_buffer);
			mob_look(mob);
			return;
		}
		ex = ex->next;
	}
	sprintf(output_buffer,"Can't go %s.\r\n", dir);
	mob_send_string(mob, output_buffer);
}

static void do_go(int playerid, char *arg)
{
	mob_go(p[playerid].object, arg);
}

static void do_up(int playerid, char *arg)
{
	do_go(playerid,"up");
}

static void do_down(int playerid, char *arg)
{
	do_go(playerid,"down");
}

static void do_north(int playerid, char *arg)
{
	do_go(playerid,"north");
}

static void do_south(int playerid, char *arg)
{
	do_go(playerid,"south");
}

static void do_east(int playerid, char *arg)
{
	do_go(playerid,"east");
}

static void do_west(int playerid, char *arg)
{
	do_go(playerid,"west");
}

static void do_northeast(int playerid, char *arg)
{
	do_go(playerid,"northeast");
}

static void do_northwest(int playerid, char *arg)
{
	do_go(playerid,"northwest");
}

static void do_southeast(int playerid, char *arg)
{
	do_go(playerid,"southeast");
}

static void do_southwest(int playerid, char *arg)
{
	do_go(playerid,"southwest");
}

/* flags:
 * 1 = takes 1 arg...
 * -1 = end of list */

static int check_commandlist(int playerid, char *command)
{
	int index;

	index = 0;
	while (commands[index].flags != -1)
	{
		if (commands[index].flags & 0x1)	/* takes arg */
		{
			if (strncmp(command,
				    commands[index].name,
				    commands[index].length) == 0)
			{
				commands[index].func(playerid,
					command+commands[index].length);
				return(1);
			}
		} else {
			if (strcmp(command,commands[index].name) == 0)
			{
				commands[index].func(playerid,"");
				return(1);
			}
		}
		index++;
	}
	return(0);
}

static void parse_input(int playerid, char *buffer, int len, int oob)
{
	char *command;
	char output_buffer[OUT_BUFFER_SIZE];

	if (oob)
	{
		sprintf(output_buffer,"\r\n%s oob condition: %s\r\n",
			p[playerid].object->name, buffer);
		say(output_buffer);
		return;
	}
	/* read until newline, or... */
	while ((*buffer != '\n') && (*buffer != '\r') &&
	/* we really filled up the buffer. */
	       (p[playerid].command_tail < READ_BUFFER_SIZE))
	{
		p[playerid].command_buffer[p[playerid].command_tail] = *buffer;
		if (*buffer == 0)
		{
			return;
		}
		p[playerid].command_tail++;
		buffer++;
	}
	/* terminate the string, if needed */
	p[playerid].command_buffer[p[playerid].command_tail] = 0;
	/* and inform the next time that there isn't a partial buffer */
	p[playerid].command_tail = 0;
	command = p[playerid].command_buffer;

	if (p[playerid].state == 1)	/* chat mode */
	{
		if (strcmp(command,".") == 0)
		{
			p[playerid].state = 0;
			sprintf(output_buffer, "Ending chat mode.\r\n");
			send_string(playerid, output_buffer);
		} else {
			sprintf(output_buffer,"\r\n%s says: %s\r\n",
				p[playerid].object->name, command);
			say(output_buffer);
		}
	}
	else if (!check_commandlist(playerid,command))
	{
		/* unrecognized command case */
		if (*command)
		{
			if (p[playerid].state == 0)
			{
				sprintf(output_buffer,"Command \"%s\" unrecognized: try \"help\".\r\n",command);
				send_string(playerid, output_buffer);
			} else {
				sprintf(output_buffer,"\r\n%s says: %s\r\n",
					p[playerid].object->name, command);
				say(output_buffer);
			}
		}
	}
}

static void end_player(int playerid)
{
	char output_buffer[OUT_BUFFER_SIZE];

	/* reinit player slot */
	p[playerid].command_buffer[0] = 0;
	p[playerid].command_tail = 0;
	p[playerid].state = -1;	/* open slot */
	p[playerid].wrap = 0;
	sprintf(output_buffer,"Bye, %d!\r\n",playerid);
	if (p[playerid].object != (struct object *)NULL)
		free_object(p[playerid].object);
	p[playerid].object = (struct object *)NULL;
	say(output_buffer);
}

static void mob_ai(struct timeval when, int data)
{
	struct timeval temp;

	mob_go(m[data].object, random_exit(m[data].object->where));
	temp.tv_sec = when.tv_sec + m[data].delay;
	temp.tv_usec = when.tv_usec;
	queue_event(temp,mob_ai,data);
}

static void config_port(int port)
{
        port_number = port;
}

static void config_opt(char *opt, char *val)
{
	long int port;
	char *endptr;

	if (!strcmp(opt,"port"))
	{
		port = strtol(val,&endptr,0);
		/* converted all of val? */
		if ((*endptr == 0) && (endptr != val))
		{
			port_number = port;
		}
	}
}

struct interface bigworld_supplies[] = {
	{ "config_port", config_port, "void(int)", NULL },
	{ "config_opt", config_opt, "void(char*,char*)", NULL },
	{NULL,NULL,NULL,NULL}
};

struct interface bigworld_uses[] = {
	{ "queue_event", NULL, "void(timeval,void(*)(timeval,int),int)", NULL},
	{ "close_connection", NULL, "void(int)", NULL },
	{ "real_connection", NULL, "int(int)", NULL },
	{ "connection_name", NULL, "char*(int)", NULL },
	{ "connection_time", NULL, "timeval(int)", NULL },
	{ "open_port", NULL, "int(int,unsigned int,int(*)(int,void(*)(int,char*,int,int)),void(*)(int),void(*)(int,char*,int,int))", NULL },
	{ "close_port", NULL, "void(int)", NULL },
	{ NULL, NULL, NULL, NULL }
};

void bigworld_use_functions(struct interface *it)
{
	struct interface *temp;

	temp = it;
	while (temp->name != NULL)
	{
		if ((!strcmp(temp->name,"queue_event")) &&
		    (!strcmp(temp->prototype,"void(timeval,void(timeval,int),int)")))
		{
			queue_event = temp->function;
		}
		else if ((!strcmp(temp->name,"close_connection")) &&
			 (!strcmp(temp->prototype,"void(int)")))
		{
			close_connection = temp->function;
		}
		else if ((!strcmp(temp->name,"real_connection")) &&
			 (!strcmp(temp->prototype,"int(int)")))
		{
			real_connection = temp->function;
		}
		else if ((!strcmp(temp->name,"connection_name")) &&
			 (!strcmp(temp->prototype,"char*(int)")))
		{
			connection_name = temp->function;
		}
		else if ((!strcmp(temp->name,"connection_time")) &&
			 (!strcmp(temp->prototype,"timeval(int)")))
		{
			connection_time = temp->function;
		}
		else if ((!strcmp(temp->name,"open_port")) &&
			 (!strcmp(temp->prototype,"int(int,unsigned int,int(*)(int,void(*)(int,char*,int,int)),void(*)(int),void(*)(int,char*,int,int))")))
		{
			open_port = temp->function;
		}
		else if ((!strcmp(temp->name,"close_port")) &&
			 (!strcmp(temp->prototype,"void(int)")))
		{
			close_port = temp->function;
		}
		temp++;
	}
	return;
}

int bigworld_initialize(void)
{
	return(0);
}

/* TODO: move some of this to initialize... */
/* TODO: add support for a go function */
/* TODO: split into modules */
int bigworld_start(void)
{
	int i;
	/* TODO: use a room loader of some sort */
	struct object *other_room, *south_room;
	struct object *slimy, *slowly, *sneaky;
	struct timeval temp;

	if ((queue_event == NULL) ||
	    (close_connection == NULL) ||
	    (real_connection == NULL) ||
	    (connection_name == NULL) ||
	    (connection_time == NULL) ||
	    (open_port == NULL) ||
	    (close_port == NULL))
	{
		fprintf(stderr,"Needed functions still uninitialized in bigworld_start!\n");
		return(-1);
	}

	open_port(port_number,INADDR_ANY,new_player,end_player,parse_input);

	root_object = new_object();
	root_object->name = strdup("root_object");

	start_room = new_object();
	start_room->name = strdup("the start room");
	insert_object(start_room, root_object);
	other_room = new_object();
	other_room->name = strdup("the other room");
	insert_object(other_room, root_object);
	south_room = new_object();
	south_room->name = strdup("the south room");
	insert_object(south_room, root_object);

	slimy = new_object();
	slimy->name = strdup("Slimy the snail");
	insert_object(slimy, other_room);
	m[0].object = slimy;
	m[0].delay = 1800;

	slowly = new_object();
	slowly->name = strdup("Slowly the slime mold");
	insert_object(slowly, other_room);
	m[1].object = slowly;
	m[1].delay = 3600;

	sneaky = new_object();
	sneaky->name = strdup("Sneaky the ninja");
	sneaky->flags |= SNEAKY_FLAG;
	insert_object(sneaky, other_room);
	m[2].object = sneaky;
	m[2].delay = 14;

	gettimeofday(&current_time,NULL);
	temp.tv_sec = current_time.tv_sec + 900;
	temp.tv_usec = current_time.tv_usec;
	queue_event(temp, mob_ai, 0);
	temp.tv_sec = current_time.tv_sec + 1800;
	temp.tv_usec = current_time.tv_usec;
	queue_event(temp, mob_ai, 1);
	temp.tv_sec = current_time.tv_sec + 7;
	temp.tv_usec = current_time.tv_usec;
	queue_event(temp, mob_ai, 2);

	link_exit("west", start_room, other_room);
	link_exit("east", other_room, start_room);
	link_exit("south", start_room, south_room);
	link_exit("north", south_room, start_room);

	/* initialize player data to known values */
	for(i=0;i<MAX_PLAYERS;i++)
	{
		p[i].command_buffer[0] = 0;
		p[i].command_tail = 0;
		p[i].state = -1;	/* open slot */
		p[i].wrap = 0;
	}
	return(0);
}

int bigworld_stop(void)
{
	return(-1);
}
