/* Example DevMUD module to exercise interface standard and socket module,
 * using a tiled graphical client */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "../include/devmud.h"

struct object {
	int x, y, dir;
};

/* TODO static number of players is very bad to assume */
#define MAX_PLAYERS 200 /* only 256 descriptors... */
#define MAX_MOBS 4
#define READ_BUFFER_SIZE 16384
#define OUT_BUFFER_SIZE (READ_BUFFER_SIZE+256)
#define LASTLOG_SIZE 10

static void (*queue_event)(struct timeval when, void (*func)(struct timeval when, int data), int data) = NULL;
static void (*close_connection)(int which) = NULL;
static int (*real_connection)(int which) = NULL;
static char *(*connection_name)(int which) = NULL;
static struct timeval (*connection_time)(int which) = NULL;
static int (*open_port)
		(int port,
		 unsigned int addr,
		 int (*new_connection)
		       (int id,
			void (*message)(int id,char *message,int len,int oob)),
		 void (*end_connection)(int id),
		 void (*message)(int id,char *message,int len,int oob))
	= NULL;
static void (*close_port)(int id) = NULL;

struct playerinfo {
	/* function to send messages */
	void (*message)(int id,char *message,int len,int oob);
	int id; /* network ID */
	char command_buffer[READ_BUFFER_SIZE];
	int command_tail;
	int x, y, xoff, yoff, newx, newy;
	int state;
} p[MAX_PLAYERS];

struct mobinfo {
	struct object *object;
	int delay;
};

static struct mobinfo m[MAX_MOBS];

static int port_number = 2002;

#define MAPX 20
#define MAPY 20
#define VMAPX 10
#define VMAPY 10

static int map[MAPX][MAPY] = {
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 1, 0, 0, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 0, 1, 0, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 0, 1 },
	{ 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1, 0, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 1 },
	{ 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1 },
	{ 1, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1 },
	{ 1, 1, 0, 0, 0, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1 },
	{ 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1 },
	{ 1, 1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1 },
	{ 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1 },
	{ 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1 },
	{ 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
};

static struct object *new_object(void)
{
	struct object *this;

	this = (struct object *)malloc(sizeof(struct object));
	if (this == NULL)
	{
		fprintf(stderr,"Fatal: Out of memory.\n");
		exit(-1);
	}

	this->x = 0;	/* no location */
	this->y = 0;	/* no location */
	this->dir = 0;	/* no direction */

	return(this);
}

static void send_string(int playerid, char *str)
{
	p[playerid].message(p[playerid].id,str,-1,0);
}

static void tile_announce(int x, int y)
{
	int loop, tx, ty, tval;
	char buffer[80];

	for (loop=0;loop<MAX_PLAYERS;loop++)
	{
		if (p[loop].state != -1)
		{
			tx = x-p[loop].xoff;
			ty = y-p[loop].yoff;
			if ((tx >= 0) &&
			    (ty >= 0) &&
			    (tx < VMAPX) &&
			    (ty < VMAPY))
			{
				if ((x == p[loop].newx) && (y == p[loop].newy))
				{
					tval = 4;
				} else {
					tval = map[x][y];
				}
				
				sprintf(buffer,"Tile %d,%d is %d\n",tx,ty,tval);
				send_string(loop,buffer);
			}
		}
	}
}

static void sendmap(int playerid)
{
	int x, y, tval;
	char buffer[80];

	for (x = 0; x<VMAPX; x++)
	{
		for (y = 0; y<VMAPX; y++)
		{
			if ((x+p[playerid].xoff == p[playerid].x) &&
			     (y+p[playerid].yoff == p[playerid].y))
			{
				tval = 4;
			} else {
				tval = map[x+p[playerid].xoff][y+p[playerid].yoff];
			}
			sprintf(buffer,"Tile %d,%d is %d\n",x,y,tval);
			send_string(playerid,buffer);
		}
	}
}

static void recenter(int playerid)
{
	int tx, ty;

	tx = p[playerid].x - VMAPX/2;
	ty = p[playerid].y - VMAPY/2;
	if (tx < 0) tx = 0;
	if (ty < 0) ty = 0;
	if (tx > MAPX-VMAPX) tx = MAPX-VMAPX;
	if (ty > MAPY-VMAPY) ty = MAPY-VMAPY;
	if ((tx != p[playerid].xoff) ||
	    (ty != p[playerid].yoff))
	{
		p[playerid].xoff = tx;
		p[playerid].yoff = ty;
		sendmap(playerid);
	}
}

static int new_player(int id,void (*message)(int id,char *message,int len,int oob))
{
	int x, y, playerid;

	/* find a playerid */
	for (playerid=0;
	     (playerid<MAX_PLAYERS) && (p[playerid].state != -1);
	     playerid++)
		;       /* do nothing */
	if (playerid >= MAX_PLAYERS)
	{
		/* TODO bail on player */
	}

	p[playerid].state = 0;
	p[playerid].message = message;
	p[playerid].id = id;
	p[playerid].command_tail = 0;
	/* find an empty space to put them */
	p[playerid].x = 255;
	p[playerid].y = 255;
	for (x = 1; (x<MAPX) && (p[playerid].x == 255); x++)
	{
		for (y = 1; (y<MAPX) && (p[playerid].x == 255); y++)
		{
			if (map[x][y] == 0)
			{
				p[playerid].x = x;
				p[playerid].y = y;
				p[playerid].newx = x;
				p[playerid].newy = y;
				p[playerid].xoff = 0;
				p[playerid].yoff = 0;
				map[x][y] = 2;
				tile_announce(x,y);
			}
		}
	}
	/* and report the whole map */
	recenter(playerid);
	sendmap(playerid);
	return(playerid);	/* what ID to use to contact us */
}

static int move_tile(int x, int y, int dx, int dy)
{
	/* legal move? */
	if (map[x+dx][y+dy] == 0)
	{
		map[x+dx][y+dy] = map[x][y];
		map[x][y] = 0;
		tile_announce(x,y);
		tile_announce(x+dx,y+dy);
		return(1);	/* happened */
	}
	return(0);	/* did not happen */
}

static int check_commandlist(int playerid, char *command)
{
	int index;

	index = 0;
	if (strncmp(command,"north",5) == 0)
	{
		p[playerid].newy--;
		if (move_tile(p[playerid].x,p[playerid].y,0,-1))
		{
			p[playerid].y -= 1;
			recenter(playerid);
		} else {
			p[playerid].newx = p[playerid].x;
			p[playerid].newy = p[playerid].y;
		}
		return(1);
	}
	if (strncmp(command,"south",5) == 0)
	{
		p[playerid].newy++;
		if (move_tile(p[playerid].x,p[playerid].y,0,1))
		{
			p[playerid].y += 1;
			recenter(playerid);
		} else {
			p[playerid].newx = p[playerid].x;
			p[playerid].newy = p[playerid].y;
		}
		return(1);
	}
	if (strncmp(command,"east",4) == 0)
	{
		p[playerid].newx++;
		if (move_tile(p[playerid].x,p[playerid].y,1,0))
		{
			p[playerid].x += 1;
			recenter(playerid);
		} else {
			p[playerid].newx = p[playerid].x;
			p[playerid].newy = p[playerid].y;
		}
		return(1);
	}
	if (strncmp(command,"west",4) == 0)
	{
		p[playerid].newx--;
		if (move_tile(p[playerid].x,p[playerid].y,-1,0))
		{
			p[playerid].x -= 1;
			recenter(playerid);
		} else {
			p[playerid].newx = p[playerid].x;
			p[playerid].newy = p[playerid].y;
		}
		return(1);
	}
	if (strncmp(command,"quit",4) == 0)
	{
		close_connection(playerid);
		return(1);
	}
	return(0);
}

static void parse_input(int playerid, char *buffer, int len, int oob)
{
	char *command;
	char output_buffer[OUT_BUFFER_SIZE];

	if (oob)
	{
		return;
	}
	/* read until newline, or... */
	while ((*buffer != '\n') && (*buffer != '\r') &&
	       /* we really filled up the buffer. */
	       (p[playerid].command_tail < READ_BUFFER_SIZE))
	{
		p[playerid].command_buffer[p[playerid].command_tail] = *buffer;
		if (*buffer == 0)
		{
			return;
		}
		p[playerid].command_tail++;
		buffer++;
	}
	/* terminate the string, if needed */
	p[playerid].command_buffer[p[playerid].command_tail] = 0;
	/* and inform the next time that there isn't a partial buffer */
	p[playerid].command_tail = 0;
	command = p[playerid].command_buffer;

	if (!check_commandlist(playerid,command))
	{
		/* unrecognized command case */
		if (*command)
		{
			sprintf(output_buffer,"Command \"%s\" unrecognized: try \"help\".\r\n",command);
			send_string(playerid, output_buffer);
		}
	}
}

static void end_player(int playerid)
{
	map[p[playerid].x][p[playerid].y] = 0;
	p[playerid].state = -1; /* open slot */
	p[playerid].command_buffer[0] = 0;
	p[playerid].command_tail = 0;
	tile_announce(p[playerid].x,p[playerid].y);
}

static void mob_ai(struct timeval when, int data)
{
	int action;
	struct timeval temp;
	struct object *mob;

	mob = m[data].object;
	action = (lrand48()%8);
	switch(action)
	{
		case 0:
			mob->dir = ((mob->dir+1) % 4);
			break;
		case 1:
			mob->dir = ((mob->dir+2) % 4);
			break;
		case 2:
			mob->dir = ((mob->dir+3) % 4);
			break;
		default:
			switch(mob->dir)
			{
				case 0:
					if (move_tile(mob->x,mob->y,0,-1))
					{
						mob->y--;
					}
					break;
				case 1:
					if (move_tile(mob->x,mob->y,1,0))
					{
						mob->x++;
					}
					break;
				case 2:
					if (move_tile(mob->x,mob->y,0,1))
					{
						mob->y++;
					}
					break;
				case 3:
					if (move_tile(mob->x,mob->y,-1,0))
					{
						mob->x--;
					}
					break;
			}
			break;
	}
	temp.tv_sec = when.tv_sec + m[data].delay;
	temp.tv_usec = when.tv_usec;
	queue_event(temp,mob_ai,data);
}

static void config_port(int port)
{
	port_number = port;
}

static void config_opt(char *opt, char *val)
{ 
	long int port;
	char *endptr;

	if (!strcmp(opt,"port"))
	{
		port = strtol(val,&endptr,0);
		/* converted all of val? */
		if ((*endptr == 0) && (endptr != val))
		{
			port_number = port;
		}
	}
}

struct interface mazeworld_supplies[] = {
	{ "config_port", config_port, "void(int)", NULL },
	{ "config_opt", config_opt, "void(char*,char*)", NULL },
	{NULL,NULL,NULL,NULL}
};

struct interface mazeworld_uses[] = {
	{ "queue_event", NULL, "void(timeval,void(*)(timeval,int),int)", NULL},
	{ "close_connection", NULL, "void(int)", NULL },
	{ "real_connection", NULL, "int(int)", NULL },
	{ "connection_name", NULL, "char*(int)", NULL },
	{ "connection_time", NULL, "timeval(int)", NULL },
	{ "open_port", NULL, "int(int,unsigned int,int(*)(int,void(*)(int,char*,int,int)),void(*)(int),void(*)(int,char*,int,int))", NULL },
	{ "close_port", NULL, "void(int)", NULL },
	{ NULL, NULL, NULL, NULL }
};

void mazeworld_use_functions(struct interface *it)
{
	struct interface *temp;

	temp = it;
	while (temp->name != NULL)
	{
		if ((!strcmp(temp->name,"queue_event")) &&
		    (!strcmp(temp->prototype,"void(timeval,void(timeval,int),int)")))
		{
			queue_event = temp->function;
		}
		else if ((!strcmp(temp->name,"close_connection")) &&
			 (!strcmp(temp->prototype,"void(int)")))
		{
			close_connection = temp->function;
		}
		else if ((!strcmp(temp->name,"real_connection")) &&
			 (!strcmp(temp->prototype,"int(int)")))
		{
			real_connection = temp->function;
		}
		else if ((!strcmp(temp->name,"connection_name")) &&
			 (!strcmp(temp->prototype,"char*(int)")))
		{
			connection_name = temp->function;
		}
		else if ((!strcmp(temp->name,"connection_time")) &&
			 (!strcmp(temp->prototype,"timeval(int)")))
		{
			connection_time = temp->function;
		}
		else if ((!strcmp(temp->name,"open_port")) &&
			 (!strcmp(temp->prototype,"int(int,unsigned int,int(*)(int,void(*)(int,char*,int,int)),void(*)(int),void(*)(int,char*,int,int))")))
		{
			open_port = temp->function;
		}
		else if ((!strcmp(temp->name,"close_port")) &&
			 (!strcmp(temp->prototype,"void(int)")))
		{
			close_port = temp->function;
		}
		temp++;
	}
	return;
}

int mazeworld_initialize(void)
{
	return(0);
}

/* TODO: move some of this to initialize... */
/* TODO: add support for a go function */
/* TODO: split into modules */
int mazeworld_start(void)
{
	int i;
	/* TODO: use a room loader of some sort */
	struct object *slimy;
	struct timeval temp;

	if ((queue_event == NULL) ||
	    (close_connection == NULL) ||
	    (real_connection == NULL) ||
	    (connection_name == NULL) ||
	    (connection_time == NULL) ||
	    (open_port == NULL) ||
	    (close_port == NULL))
	{
		fprintf(stderr,"Needed functions still uninitialized in mazeworld_start!\n");
		return(-1);
	}

	open_port(port_number,INADDR_ANY,new_player,end_player,parse_input);

	slimy = new_object();
	slimy->x = 4;
	slimy->y = 4;
	slimy->dir = 0;
	map[slimy->x][slimy->y] = 3;
	gettimeofday(&current_time,NULL);
	m[0].object = slimy;
	m[0].delay = 2;
	temp.tv_sec = current_time.tv_sec + 2;
	temp.tv_usec = current_time.tv_usec;
	queue_event(temp, mob_ai, 0);

	/* initialize player data to known values */
	for(i=0;i<MAX_PLAYERS;i++)
	{
		p[i].command_buffer[0] = 0;
		p[i].command_tail = 0;
		p[i].x = 0;
		p[i].y = 0;
		p[i].newx = 0;
		p[i].newy = 0;
		p[i].state = -1;
	}
	return(0);
}

int mazeworld_stop(void)
{
	return(-1);
}
