/* Event and FD manager (select) -- a general main loop module.
 *
 * It should accept (fd,function,id) and (time,function,id) tuples.
 * when fd is ready, call the appropriate function with that id.
 * when time occurs, call the appropriate function with that id.
 * Send the fd/time too, I think...
 *
 * event cancel should be supported but slow? -- canceled event bit.
 * objects should count outstanding events, delete is delayed.
 * fd cancel should work.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <errno.h>
#include "../include/devmud.h"

#define MAX_FDLIST FD_SETSIZE	/* get real value from sys/types */

struct fdnotify {
	int fd;
	void (*func)(int fd, int data);
	int data;
};

struct event {
	struct timeval when;
	void (*func)(struct timeval when, int data);
	int data;
};

struct heap {
	struct event *events;
	int num_events, count;
};

static struct heap h;
static struct fdnotify readfdlist[MAX_FDLIST],
		       writefdlist[MAX_FDLIST],
		       exceptfdlist[MAX_FDLIST];
static int readfdcount = 0,
	   writefdcount = 0,
	   exceptfdcount = 0;
static int done = 0;

extern int errno;

/* check for dups? */
static void register_readfd(int fd, void (*func)(int fd, int data), int data)
{
	readfdlist[readfdcount].fd = fd;
	readfdlist[readfdcount].func = func;
	readfdlist[readfdcount].data = data;
	readfdcount++;
}

static void register_writefd(int fd, void (*func)(int fd, int data), int data)
{
	writefdlist[writefdcount].fd = fd;
	writefdlist[writefdcount].func = func;
	writefdlist[writefdcount].data = data;
	writefdcount++;
}

static void register_exceptfd(int fd, void (*func)(int fd, int data), int data)
{
	exceptfdlist[exceptfdcount].fd = fd;
	exceptfdlist[exceptfdcount].func = func;
	exceptfdlist[exceptfdcount].data = data;
	exceptfdcount++;
}

static void unregister_readfd(int fd)
{
	int i;

	for (i=0; i<readfdcount; i++)
	{
		if (readfdlist[i].fd == fd)
		{
			readfdcount--;
			readfdlist[i] = readfdlist[readfdcount];
			return;
		}
	}
	/* log error or something? -- no such FD registered */
}

static void unregister_writefd(int fd)
{
	int i;

	for (i=0; i<writefdcount; i++)
	{
		if (writefdlist[i].fd == fd)
		{
			writefdcount--;
			writefdlist[i] = writefdlist[writefdcount];
			return;
		}
	}
	/* log error or something? -- no such FD registered */
}

static void unregister_exceptfd(int fd)
{
	int i;

	for (i=0; i<exceptfdcount; i++)
	{
		if (exceptfdlist[i].fd == fd)
		{
			exceptfdcount--;
			exceptfdlist[i] = exceptfdlist[exceptfdcount];
			return;
		}
	}
	/* log error or something? -- no such FD registered */
}

static int getmaxfdcount(void)
{
	return(MAX_FDLIST);
}

static int getreadfdcount(void)
{
	return(readfdcount);
}

static int getwritefdcount(void)
{
	return(writefdcount);
}

static int getexceptfdcount(void)
{
	return(exceptfdcount);
}

/* Slight bias towards doing early fd's first...  Oh well. */
static void try_select(struct timeval *timeout)
{
	int i, result, nfds;
	fd_set readfds, writefds, exceptfds;

	nfds = 0;
	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_ZERO(&exceptfds);
	for (i=0; i<readfdcount; i++)
	{
		FD_SET(readfdlist[i].fd, &readfds);
		if (readfdlist[i].fd >= nfds)
			nfds = readfdlist[i].fd+1;
	}
	for (i=0; i<writefdcount; i++)
	{
		FD_SET(writefdlist[i].fd, &writefds);
		if (writefdlist[i].fd >= nfds)
			nfds = writefdlist[i].fd+1;
	}
	for (i=0; i<exceptfdcount; i++)
	{
		FD_SET(exceptfdlist[i].fd, &exceptfds);
		if (exceptfdlist[i].fd >= nfds)
			nfds = exceptfdlist[i].fd+1;
	}
	result = select(nfds, &readfds, &writefds, &exceptfds, timeout);
	if (result == -1)
	{
		if (errno != EINTR)
		{
			perror("select");
		}
		return;
	}
	else if (result == 0)
	{
		return;
	}
	for (i=0; i<readfdcount; i++)
	{
		if (FD_ISSET(readfdlist[i].fd, &readfds))
		{
			/* notify the function */
			readfdlist[i].func(readfdlist[i].fd,
					   readfdlist[i].data);
		}
	}
	for (i=0; i<writefdcount; i++)
	{
		if (FD_ISSET(writefdlist[i].fd, &writefds))
		{
			/* notify the function */
			writefdlist[i].func(writefdlist[i].fd,
					    writefdlist[i].data);
		}
	}
	for (i=0; i<exceptfdcount; i++)
	{
		if (FD_ISSET(exceptfdlist[i].fd, &exceptfds))
		{
			/* notify the function */
			exceptfdlist[i].func(exceptfdlist[i].fd,
					     exceptfdlist[i].data);
		}
	}
}

static void queue_event(struct timeval when, void (*func)(struct timeval when, int data), int data)
{
	unsigned int loc, newloc;
	struct event tempevent, *new_events;

	if (h.count == h.num_events)
	{
		h.num_events *= 2;

		/* get more memory */
		new_events = (struct event *)
			realloc(h.events,(h.num_events)*sizeof(struct event));

		/* and check it */
		if (new_events == (struct event *)NULL)
		{
			fprintf(stderr,"Couldn't realloc event events.\n");
			/* put it back... */
			h.num_events /= 2;
			exit(1);
		}
		h.events = new_events;
	}
	loc = h.count++;
	h.events[loc].when = when;
	h.events[loc].func = func;
	h.events[loc].data = data;
	/* event at loc may now violate the heap property... */

	newloc = (loc-1)/2;
	while ((loc > 0) &&
	       ((h.events[loc].when.tv_sec < h.events[newloc].when.tv_sec) ||
		((h.events[loc].when.tv_sec == h.events[newloc].when.tv_sec) &&
		 (h.events[loc].when.tv_usec < h.events[newloc].when.tv_usec))))
	{
		tempevent = h.events[loc];
		h.events[loc] = h.events[newloc];
		h.events[newloc] = tempevent;
		loc = newloc;
		newloc = (loc-1)/2;
	}
}

/* remove the top heap event */
static void delete_event(void)
{
	struct event tempevent;
	int loc, newloc, end;

	if (h.count == 0)
	{
		/* heap is empty -- can't get an event */
		return;
	}

	h.count--;
	end = h.count;		/* last valid spot */
	/* stuff the last event at the top -- we'll fix it shortly */
	h.events[0] = h.events[end];
	end--;			/* that spot is no longer valid */
	loc = 0;

	while (1)	/* return from the loop when it's ok */
	{
		newloc = (loc * 2) + 1;
		if (newloc >= end)		/* < 2 children */
		{
			if ((newloc == end) &&
			    ((h.events[newloc].when.tv_sec <
					h.events[loc].when.tv_sec) ||
			     ((h.events[newloc].when.tv_sec ==
					h.events[loc].when.tv_sec) &&
			      (h.events[newloc].when.tv_usec ==
					h.events[loc].when.tv_usec))))
			{
				tempevent = h.events[loc];
				h.events[loc] = h.events[newloc];
				h.events[newloc] = tempevent;
			}
			return;
		}
		/* now known to have 2 children */
		if ((h.events[newloc].when.tv_sec >
			h.events[newloc+1].when.tv_sec) ||
		    ((h.events[newloc].when.tv_sec ==
			h.events[newloc+1].when.tv_sec) &&
		     (h.events[newloc].when.tv_usec >
			h.events[newloc+1].when.tv_usec)))
		{
			newloc++;
		}
		if ((h.events[newloc].when.tv_sec <
			h.events[loc].when.tv_sec) ||
		    ((h.events[newloc].when.tv_sec ==
			h.events[loc].when.tv_sec) &&
		     (h.events[newloc].when.tv_sec <
			h.events[loc].when.tv_sec)))
		{
				tempevent = h.events[loc];
				h.events[loc] = h.events[newloc];
				h.events[newloc] = tempevent;
				loc = newloc;
		} else {
			return;
		}
	}
}

struct interface select_supplies[] = {
	{ "register_readfd", register_readfd, "void(int,void(int,int),int)", NULL },
	{ "register_writefd", register_writefd, "void(int,void(int,int),int)", NULL },
	{ "register_exceptfd", register_exceptfd, "void(int,void(int,int),int)", NULL },
	{ "unregister_readfd", unregister_readfd, "void(int)", NULL },
	{ "unregister_writefd", unregister_writefd, "void(int)", NULL },
	{ "unregister_exceptfd", unregister_exceptfd, "void(int)", NULL },
	{ "queue_event", queue_event,
		"void(timeval,void(timeval,int),int)", NULL },
	{ "getmaxfdcount", getmaxfdcount, "int(void)", NULL },
	{ "getreadfdcount", getreadfdcount, "int(void)", NULL },
	{ "getwritefdcount", getwritefdcount, "int(void)", NULL },
	{ "getexceptfdcount", getexceptfdcount, "int(void)", NULL },
	{ NULL, NULL, NULL, NULL }
};

struct interface select_uses[] = {
	{ NULL, NULL, NULL, NULL }
};

void select_use_functions(struct interface *it)
{
	return;
}

int select_initialize(void)
{
	/* set up the heap for events */
	h.count = 0;
	h.num_events = 32;	/* arbitrary number of starting events */
	/* get the memory */
	h.events = (struct event *)
			malloc((h.num_events)*sizeof(struct event));
	/* and check it */
	if (h.events == (struct event *)NULL)
	{
		fprintf(stderr,"Couldn't allocate memory for heap events.\n");
		return(-1);
	}
	return(0);
}

int select_start(void)
{
	struct timeval delay;
	struct event current;

	gettimeofday(&current_time,NULL);	/* init idea of time */

	while(!done)
	{
		if (h.count == 0)
		{
			/* wait indefinitely */
			try_select(NULL);
		} else {
			gettimeofday(&current_time,NULL);
			delay.tv_sec = h.events[0].when.tv_sec - current_time.tv_sec;
			delay.tv_usec = h.events[0].when.tv_usec - current_time.tv_usec;
			if (delay.tv_usec < 0)  /* normalize */
			{
				delay.tv_usec += 1000000;
				delay.tv_sec--;
			}
			/* time for which calling select isn't worth it? */
			if ((delay.tv_sec > 0) ||
			    ((delay.tv_sec == 0) && (delay.tv_usec > 0)))
			{
				try_select(&delay);
			}
		}
		while ((h.count != 0) &&
		       ((current_time.tv_sec > h.events[0].when.tv_sec) ||
			((current_time.tv_sec == h.events[0].when.tv_sec) &&
			 (current_time.tv_usec >= h.events[0].when.tv_usec)))) {
			current = h.events[0];
			delete_event();
			(*current.func)(current.when,current.data);
		}
	}
	/* clean up the heap */
	free(h.events);
	return(0);
}

int select_stop(void)
{
	done = 1;
	return(0);	/* will stop */
}

