#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#define TELOPTS
#include <arpa/telnet.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include "../include/devmud.h"

/* limited by number of file descriptors */
#define MAX_CONNECTIONS 200

#define BACKLOG		5	/* number of pending conections.  OS limit */
#define MAXHOSTNAME	32	/* longest hostname dealt with. */
#define MAX_PACKET	512	/* with header should fit in 576 minimum */
#define FULL_MESSAGE	"All connections are taken.\n"
#define SITE_NAME_SIZE	80

#define ECHO_OFF	0
#define ECHO_ON		1
#define ECHO_OFF_SNT	2
#define ECHO_ON_SNT	3
#define ECHO_OFF_REV	4
#define ECHO_ON_REV	5

struct interactive
{
	int socket, listenindex, mid, echostate, parsestate;
	struct timeval last_time;
	struct sockaddr_in addr;
	char site_name[SITE_NAME_SIZE];
};

struct listensocket
{
	int port, count, closed;
	int (*new_connection) (int id,
			       void (*message)(int id,char *message,int len,int oob));
	void (*end_connection)(int id);
	void (*message)(int id,char *message,int len,int oob);
};

extern int errno;
static struct interactive *connections[MAX_CONNECTIONS];
static struct listensocket *listeners[MAX_CONNECTIONS];
static int connectionscount = 0;
static int listensocketcount = 0;

/* functions this module needs */
static void (*register_readfd)(int fd, void (*func)(int fd, int data), int data) = NULL;
static void (*unregister_readfd)(int fd) = NULL;

static void free_listener(int which)
{
	listeners[which] = (struct listensocket *)NULL;
	while ((listensocketcount > 0) &&
	       (listeners[listensocketcount-1] == (struct listensocket *)NULL))
	{
		listensocketcount--;
	}
}

static void close_connection(int which)
{
	int lid;

	if (connections[which] == NULL)
	{
		fprintf(stderr,"Attempt to close a non connection.\n");
		return;
	}
	if (shutdown(connections[which]->socket, 2) == -1)
	{
		perror("shutdown socket");
	}
	if (close(connections[which]->socket) == -1)
	{
		perror("close socket");
	}
	unregister_readfd(connections[which]->socket);
	lid = connections[which]->listenindex;
	listeners[lid]->end_connection(connections[which]->mid);
	listeners[lid]->count--;
	if ((listeners[lid]->count == 0) && listeners[lid]->closed)
	{
		free_listener(lid);
	}
	free(connections[which]);
	connections[which] = NULL;
	while ((connectionscount > 0) &&
	       (connections[connectionscount-1] == (struct interactive *)NULL))
	{
		connectionscount--;
	}
}

static char echo_on[3] = { IAC, WONT, TELOPT_ECHO };
static char echo_off[3] = { IAC, WILL, TELOPT_ECHO };

static void read_callback(int fd, int data)
{
	int length;
	unsigned char buffer[2048], *in, *out, reply[3];
	char *errmsg;

	/* initialize to no message recieved */
	*buffer = 0;
	if ((length = read(fd, buffer, sizeof(buffer) - 1)) == -1)
	{
		/* read error */
		if (errno < sys_nerr)
		{
			sprintf(buffer, "read error: %s", sys_errlist[errno]);
			errmsg = buffer;
		} else {
			errmsg = "Unknown read error";
		}
		perror("socket read");
		/* Out of band notification */
		listeners[connections[data]->listenindex]->message(connections[data]->mid,errmsg,strlen(errmsg),1);
		/* socket no longer exists, do cleanup */
		close_connection(data);
	}
	else if (length == 0)
	{
		errmsg = "Socket read EOF";
		/* Out of band notification */
		listeners[connections[data]->listenindex]->message(connections[data]->mid,errmsg,strlen(errmsg),1);
		close_connection(data);
	}
	else /* length > 0 */
	{
		in = buffer;
		out = buffer;
		while (in < buffer + length)
		{
			switch (connections[data]->parsestate)
			{
				case 0:
					if (*in == IAC)
					{
						connections[data]->parsestate = IAC;
					} else {
						*out++ = *in++;
					}
					break;
				case IAC:
					switch (*in)
					{
						case IAC:
							*out++ = *in++;
							connections[data]->parsestate = 0;
							break;
						case WILL:
						case WONT:
						case DO:
						case DONT:
							connections[data]->parsestate = *in;
							break;
						default:
							fprintf(stderr,"Unrecognized telnet negotiation %d.\n",*in);
							connections[data]->parsestate = 0;
							break;
						*out++ = *in++;
					}
					break;
				case WILL:
					/* don't know what you're offering,
					 * so we'll ask you not to. */
					reply[0] = IAC;
					reply[1] = DONT;
					reply[2] = *in;
					write(fd,reply,3);
					connections[data]->parsestate = 0;
					break;
				case WONT:
					/* I don't know what you're refusing,
					 * so we'll ignore you (rfc ignorer!) */
					connections[data]->parsestate = 0;
					break;
				case DO:
					if (*in == TELOPT_ECHO)
					{
						switch (connections[data]->echostate)
						{
							case ECHO_OFF:
								/* no-op */
								break;
							case ECHO_ON:
								/* refuse */
								write(fd,echo_on,3);
								break;
							case ECHO_OFF_SNT:
								/* yay! */
								connections[data]->parsestate = ECHO_OFF;
								break;
							case ECHO_ON_SNT:
								/* refused */
								connections[data]->parsestate = ECHO_OFF;
								break;
							case ECHO_OFF_REV:
								write(fd,echo_on,3);
								connections[data]->parsestate = ECHO_ON_SNT;
								break;
							case ECHO_ON_REV:
								/* yay! */
								connections[data]->parsestate = ECHO_OFF;
								break;
							default:
								fprintf(stderr,"Telnet option state is terminally confused.\n");
								break;
						}
					} else {
						/* decline with IAC WONT (x) */
						reply[0] = IAC;
						reply[1] = WONT;
						reply[2] = *in;
						write(fd,reply,3);
					}
					connections[data]->parsestate = 0;
					break;
				case DONT:
					if (*in == TELOPT_ECHO)
					{
						switch (connections[data]->echostate)
						{
							case ECHO_OFF:
								/* insisted! */
								write(fd,echo_on,3);
								connections[data]->parsestate = ECHO_ON;
								break;
							case ECHO_ON:
								/* no-op */
								break;
							case ECHO_OFF_SNT:
								/* refused */
								connections[data]->parsestate = ECHO_ON;
								break;
							case ECHO_ON_SNT:
								/* yay! */
								connections[data]->parsestate = ECHO_ON;
								break;
							case ECHO_OFF_REV:
								/* umm..  Ok */
								connections[data]->parsestate = ECHO_ON;
								break;
							case ECHO_ON_REV:
								/* yay! */
								write(fd,echo_off,3);
								connections[data]->parsestate = ECHO_OFF_SNT;
								break;
							default:
								fprintf(stderr,"Telnet option state is terminally confused.\n");
								break;
						}
					} else {
						/* RFC says Ignore it. */
					}
					connections[data]->parsestate = 0;
					break;
				default:
					fprintf(stderr,"Telnet option state is terminally confused.\n");
					break;
			}
		}
		buffer[length] = 0;
		connections[data]->last_time = current_time;
		listeners[connections[data]->listenindex]->message(connections[data]->mid,buffer,length,0);
	}
}

/* note that sending IAC is impolite in a telnet context, and the next
 * level driver should prevent that.  A number of other characters are
 * impolite as well */
/* message() eliminates IACs in this implementation */
static void put_string(int to, char *message, int length)
{
	int n;
	unsigned char buffer[2048];
	char *errmsg;

	if (connections[to] == NULL)
	{
		return;
	}
	while(length>0)
	{
		n = write(connections[to]->socket, message, length > MAX_PACKET ? MAX_PACKET : length);
		if (n == -1)
		{
			/* write error */
			if (errno < sys_nerr)
			{
				sprintf(buffer, "write error: %s", sys_errlist[errno]);
				errmsg = buffer;
			} else {
				errmsg = "Unknown write error";
			}
			perror("socket write");
			/* Out of band notification */
			listeners[connections[to]->listenindex]->message(connections[to]->mid,errmsg,strlen(errmsg),1);
			/* socket no longer exists, do cleanup */
			close_connection(to);
			return;	/* don't loop back to here in while loop */
		} else if (n == 0) {
			fprintf(stderr,"Write: wrote no data!\n");
			errmsg = "Write: wrote no data!";
			/* Out of band notification */
			listeners[connections[to]->listenindex]->message(connections[to]->mid,errmsg,strlen(errmsg),1);
			return;	/* don't loop back to here in while loop */
		} else {
			length -= n;
			message += n;
		}
	}
}

/* Telnet negotiation stuff
 * We only care about echo, but have to handle all telnet options.
 *
 * turn on echo: { IAC, WONT, TELOPT_ECHO, 0 }   We expect DONT ECHO response.
 * turn off echo: { IAC, WILL, TELOPT_ECHO, 0 }   We expect DO ECHO response.
 * Only do it if it's real, and also we're not closing it.
 * echo state machine:
 *                on = WONT       off = WILL      got_on = DONT   got_off = DO
 *        OFF     "on" ON_SNT     -               "on" ON         -
 *        ON      -               "off" OFF_SNT   -               "on" ON
 *        OFF_SNT OFF_REV         -               ON              OFF
 *        ON_SNT  -               ON_REV          ON              OFF
 *        OFF_REV -               OFF_SNT         ON              "on" ON_SNT
 *        ON_REV  ON_SNT          -               "off" OFF_SNT   OFF
 * 
 * DONT ECHO in ECHO_OFF: send echo_on -> ECHO_ON: they insisted ...
 *	for RFC compliance rather than a belief that it will happen or matter.
 * DONT ECHO in ECHO_ON:
 *	we're already not doing that
 * DONT ECHO in ECHO_OFF_SNT: -> ECHO_ON
 *	they refused to turn off local echoing
 * DONT ECHO in ECHO_ON_SNT: -> ECHO_ON
 *	they're acknowleging our request.  Yay!
 * DONT ECHO in ECHO_OFF_REVERSED: -> ECHO_ON
 *	they refused to turn off local echoing, and
 *	we've changed our mind since then anyway
 * DONT ECHO in ECHO_ON_REVERSED: send echo_off, -> ECHO_OFF_SNT
 *	thanks, but we've changed our mind
 * 
 * DO ECHO in ECHO_OFF:
 *	we're already doing that
 * DO ECHO in ECHO_ON: send echo_on
 *	they asked ... we'll decline.
 * DO ECHO in ECHO_OFF_SNT: -> ECHO_OFF
 *	they're acknowleging our request.  Yay!
 * DO ECHO in ECHO_ON_SNT: -> ECHO_OFF
 *	they refused to turn on local echoing.  Bad them.
 * DO ECHO in ECHO_OFF_REVERSED: send echo_on -> ECHO_ON_SNT
 *	thanks, but we've changed our mind
 * DO ECHO in ECHO_ON_REVERSED: -> ECHO_OFF
 *	they refused to turn on local echoing (bad),
 *	but we've changed our mind since then anyway
 * 
 * DONT (x): RFC says Ignore it.
 *	They probably shouldn't send it anyway...
 * DO (x): decline with IAC WONT (x)
 *	We don't know how to do that.
 * WONT (x): do nothing.
 *	I don't know what you're refusing, so we'll ignore you.
 *	(you rfc ignorer!)
 * WILL (x): ask them not to with IAC DONT (x)
 *	I don't know what you're talking about, so we'll ask you not to.
 */

/* length -1 means call strlen to find it. */
static void message(int to, char *message, int length, int oob)
{
	static char buffer[1024];
	char *current;

	if (oob)
	{
		if (strcmp(message,"echo on") == 0)
		{
			switch(connections[to]->echostate)
			{
				case ECHO_OFF:
					put_string(to,echo_on,3);
					connections[to]->echostate = ECHO_ON_SNT;
					break;
				case ECHO_ON:
					/* already doing that */
					break;
				case ECHO_OFF_SNT:
					connections[to]->echostate = ECHO_OFF_REV;
					break;
				case ECHO_ON_SNT:
					/* already asked to do that */
					break;
				case ECHO_OFF_REV:
					/* already want to ask to do that */
					break;
				case ECHO_ON_REV:
					connections[to]->echostate = ECHO_ON_SNT;
					break;
				default:
					fprintf(stderr,"Echo state is terminally confused.\n");
					break;
			}
		}
		else if (strcmp(message,"echo off") == 0)
		{
			switch(connections[to]->echostate)
			{
				case ECHO_OFF:
					/* already doing that */
					break;
				case ECHO_ON:
					put_string(to,echo_off,3);
					connections[to]->echostate = ECHO_OFF_SNT;
					break;
				case ECHO_OFF_SNT:
					/* already asked to do that */
					break;
				case ECHO_ON_SNT:
					connections[to]->echostate = ECHO_ON_REV;
					break;
				case ECHO_OFF_REV:
					connections[to]->echostate = ECHO_OFF_SNT;
					break;
				case ECHO_ON_REV:
					/* already want to ask to do that */
					break;
				default:
					fprintf(stderr,"Echo state is terminally confused.\n");
					break;
			}
		}
		return;	/* no other layer to pass oob messages to: lose it */
	}
	current = buffer;

	if (length == -1)
		length = strlen(message);
	while (length-- > 0)
	{
		if (*message == -1)
		{
			*current++ = -1;
		}
		*current++ = *message++;
		if (current-buffer > 1020)
		{
			*current = 0;
			put_string(to,buffer,current-buffer);
			current = buffer;
		}
	}

	if (current != buffer)
	{
		put_string(to,buffer,current-buffer);
	}
}

static void listener_callback(int fd, int data)
{
	int length, i, newfd, newspot;
	struct sockaddr_in sa;
	char temp[80];

	/* get a new socket */
	newspot = -1;
	length = sizeof(sa);
	newfd = accept(fd,(struct sockaddr *)&sa, &length);
	if (newfd != -1)
	{
		for (i=0; i<connectionscount; i++)
		{
			if (!connections[i])
			{
				newspot = i;
				break;
			}
		}
		if ((newspot == -1) && (connectionscount < MAX_CONNECTIONS))
		{
			newspot = connectionscount++;
			connections[newspot] = (struct interactive *)NULL;
		}
		if (newspot != -1)
		{
			if ((connections[newspot] = (struct interactive *)malloc(sizeof(struct interactive))) == NULL)
			{
				fprintf(stderr,"malloc failed.\n");
				exit(2);
			}
			connections[newspot]->socket = newfd;
			connections[newspot]->last_time = current_time;
			length = sizeof(struct sockaddr_in);
			getpeername(newfd, (struct sockaddr*)&connections[newspot]->addr, &length);
			connections[newspot]->site_name[0] = 0;
			strcpy(connections[newspot]->site_name,inet_ntoa(connections[newspot]->addr.sin_addr));
			sprintf(temp, " port %d",ntohs(connections[newspot]->addr.sin_port));
			strcat(connections[newspot]->site_name,temp);
			register_readfd(newfd,read_callback,newspot);

			connections[newspot]->echostate = ECHO_ON;
			connections[newspot]->parsestate = 0;
			connections[newspot]->listenindex = data;
			connections[newspot]->mid = listeners[data]->new_connection(newspot,message);
			return;
		}
		/* didn't find an open slot */
		write(newfd, FULL_MESSAGE, strlen(FULL_MESSAGE));
		close(newfd);
	}
	else
	{
		switch(errno)
		{
			case EINTR:
				break;
				return;
			default:
				perror("accept");
		}
	}
}

static int open_port(int port,
		     unsigned int addr,
		     int (*new_connection)
			 (int id,
			  void (*message)(int id,char *message,int len,int oob)),
		     void (*end_connection)(int id),
		     void (*message)(int id,char *message,int len,int oob))
{
	int parameter, descriptor, newspot, i;
	struct sockaddr_in sa;
	struct hostent *hp;
	char localhost[MAXHOSTNAME+1];

	newspot = -1;
	for (i=0; i<listensocketcount; i++)
	{
		if (!listeners[i])
		{
			newspot = i;
			break;
		}
	}
	if ((newspot == -1) && (listensocketcount < MAX_CONNECTIONS))
	{
		newspot = listensocketcount++;
		listeners[newspot] = (struct listensocket *)NULL;
	}
	if (newspot == -1)
	{
		fprintf(stderr,"All connections are full; can't open new main socket\n.");
		return(-1);
	}

	/* get our host name */
	if (gethostname(localhost, MAXHOSTNAME) == -1)
	{
		perror("Couldn't get hostname");
		return(-1);
	}

	/* get host information from the hostname */
	if ((hp = gethostbyname(localhost)) == NULL)
	{
		perror("Couldn't get local host info");
		return(-1);
	}

	/* set up our local address */
	/* this involves setting all of the fields in a sockaddr_in: */
	/* sin_family, sin_port, sin_addr, and sin_zero */

	/* this gets the sin_zero (reserved space) */
	memset((char *)&sa, 0, sizeof(sa));	/* to be safe */

	/* set the family */
	sa.sin_family = hp->h_addrtype;

	/* set the port.  This needs to be in network byte-order */
	sa.sin_port = htons(port);

	/* and set the address */
	/* let user specify -- loopback, INADDR_ANY, multiple host addrs? */
	sa.sin_addr.s_addr = addr; /* usually INADDR_ANY */

	/* create the socket */
	/* get a internet-based, two-way stream, protocol IP=0 */
	descriptor = socket(PF_INET,SOCK_STREAM,0);
	if (descriptor == -1)
	{
		perror("Couldn't get socket");
		return(-1);
	}

	/* set the socket options to allow multiple sessions */
	parameter = 1;	/* 1 = turn on */
	if (setsockopt(descriptor, SOL_SOCKET, SO_REUSEADDR,
		       (char *) &parameter, sizeof(parameter)) < 0)
	{
		perror("Couldn't set socket option");
		close(descriptor);
		descriptor = -1;
		return(-1);
	}

	/* bind the socket to an address */
	if (bind(descriptor, (struct sockaddr *)&sa, sizeof(sa)) < 0)
	{
		perror("Couldn't bind socket");
		close(descriptor);
		descriptor = -1;
		return(-1);
	}

	/* setup to accept connections */
	if (listen(descriptor, BACKLOG) == -1)
	{
		perror("Couldn't setup accepting connections");
		close(descriptor);
		descriptor = -1;
		return(-1);
	}

	if ((listeners[newspot] = (struct listensocket *)malloc(sizeof(struct listensocket))) == NULL)
	{
		fprintf(stderr,"malloc failed.\n");
		exit(2);
	}
	listeners[newspot]->port = port;
	listeners[newspot]->count = 0;
	listeners[newspot]->closed = 0;
	listeners[newspot]->new_connection = new_connection;
	listeners[newspot]->end_connection = end_connection;
	listeners[newspot]->message = message;
	register_readfd(descriptor,listener_callback,newspot);
	return(newspot);	/* id for future reference */
}

static void close_port(int id)
{
	/* close the socket */
	close(listeners[id]->port);
	unregister_readfd(listeners[id]->port);
	if (listeners[id]->count == 0)
	{
		free_listener(id);
	} else {
		listeners[id]->closed = 1;
	}
}

static int real_connection(int which)
{
	if ((which < 0) || (which >= connectionscount))
		return(0);
	return(connections[which] != NULL);
}

static struct timeval connection_time(int which)
{
	struct timeval empty;

	empty.tv_sec = 0;
	empty.tv_usec = 0;
	if ((which < 0) || (which >= connectionscount))
		return(empty);
	if (connections[which] != NULL)
		return(connections[which]->last_time);
	else
		return(empty);
}

static char *connection_name(int which)
{
	if ((which < 0) || (which >= connectionscount))
		return ("-");
	if (connections[which] != NULL)
		return(connections[which]->site_name);
	else
		return ("-");
}

struct interface socket_supplies[] = {
	{ "close_connection", close_connection, "void(int)", NULL },
	{ "real_connection", real_connection, "int(int)", NULL },
	{ "connection_name", connection_name, "char*(int)", NULL },
	{ "connection_time", connection_time, "timeval(int)", NULL },
	{ "open_port", open_port, "int(int,unsigned int,int(*)(int,void(*)(int,char*,int,int)),void(*)(int),void(*)(int,char*,int,int))", NULL },
	{ "close_port", close_port, "void(int)", NULL },
	{ NULL, NULL, NULL, NULL }
};

struct interface socket_uses[] = {
        { "register_readfd", NULL, "void(int,void(int,int),int)", NULL },
        { "unregister_readfd", NULL, "void(int)", NULL },
	{ NULL, NULL, NULL, NULL }
};

void socket_use_functions(struct interface *it)
{
        struct interface *temp;

        temp = it;
        while (temp->name != NULL)
        {
                if ((!strcmp(temp->name,"register_readfd")) &&
                    (!strcmp(temp->prototype,"void(int,void(int,int),int)")))
                {
                        register_readfd = temp->function;
                }
                else if ((!strcmp(temp->name,"unregister_readfd")) &&
                         (!strcmp(temp->prototype,"void(int)")))
                {
                        unregister_readfd = temp->function;
                }
                temp++;
        }
        return;
}

int socket_initialize(void)
{
	return(0);
}

int socket_start(void)
{
        if ((register_readfd == NULL) ||
            (unregister_readfd == NULL))
        {
                fprintf(stderr,"Needed functions still uninitialized in socket_start!\n");
                return(-1);
        }
	return(0);
}

int socket_stop(void)
{
	return(-1);
}
